# -*- coding: utf-8 -*-
import logging

from django.conf import settings
from django_yauth.authentication_mechanisms.tvm import Mechanism as BaseMechanism

from events.common_app.directory import CachedDirectoryClient
from events.common_app.tvm2_client import get_tvm2_client
from events.yauth_contrib.user import (
    FormsTVM2YandexUser,
    FormsTVM2YandexServiceUser,
    FormsAnonymousYandexUser,
)
from events.yauth_contrib.mechanisms.base import BaseFormsMixin

logger = logging.getLogger(__name__)


class Mechanism(BaseFormsMixin, BaseMechanism):
    YandexUser = FormsTVM2YandexUser
    YandexServiceUser = FormsTVM2YandexServiceUser
    AnonymousYandexUser = FormsAnonymousYandexUser

    def __init__(self):
        pass

    def extract_params(self, request):
        params = super().extract_params(request)
        if params is not None:
            params['request'] = request
            params['uid'] = request.META.get(settings.YAUTH_UID_HEADER)
            params['cloud_uid'] = request.META.get(settings.YAUTH_CLOUD_UID_HEADER)
            params['org_id'] = request.orgs[0] if request.orgs else None
        return params

    def apply(self, request, service_ticket, user_ip, user_ticket=None, uid=None, cloud_uid=None, org_id=None):
        self.tvm = self.get_tvm2_client()
        yauser = super(Mechanism, self).apply(
            service_ticket=service_ticket,
            user_ip=user_ip,
            user_ticket=user_ticket,
        )

        if isinstance(yauser, self.YandexServiceUser):
            # разрешаем ходить в бекенд с сервисными тикетами и передавать uid, как заголовок
            if uid is not None:
                if yauser.service_ticket.src in settings.YAUTH_TVM2_SERVICE_CLIENTS:
                    yauser.uid = uid

            yauser.cloud_uid = None
            if cloud_uid is not None:
                if not org_id:
                    client = CachedDirectoryClient()
                    request.orgs = [
                        str(org['id'])
                        for org in client.get_organizations(uid, cloud_uid=cloud_uid)
                    ]
                    logger.info('Got organizations %s for cloud user %s', request.orgs, cloud_uid)
                    org_id = request.orgs[0] if request.orgs else None

                if org_id:
                    yauser.cloud_uid = cloud_uid

            if not yauser.uid and not yauser.cloud_uid:
                yauser = self._make_anonymous(yauser)

        if isinstance(yauser, (self.YandexUser, self.YandexServiceUser)):
            yauser.orgs = request.orgs

        return yauser

    def get_tvm2_client(self):
        return get_tvm2_client()

    def _make_anonymous(self, yauser):
        anonymous_yauser = self.anonymous()
        anonymous_yauser.service_ticket = yauser.service_ticket
        return anonymous_yauser
