# coding: utf-8
from http.cookies import SimpleCookie
from urllib.parse import quote

import blackbox

from .app import get_destination_url
from .responses import Redirect302Response, InternalErrorResponse
from .settings import SESSIONID_COOKIE_NAME, LOGIN_URL, COOKIE_RENEW_URL, BLACKBOX_URL, \
    WHITE_LIST
from .logger import logger


# Значение атрибута ttl у короткоживущей сессии для OTP
SHORT_LIVED_SESSION_TYPE_TTL = 4

NO_AUTH_HANDLES = {
    '/ping',
    '/favicon.ico',
}


class AuthentificationError(Exception):
    pass


class NotAuthentificatedError(AuthentificationError):
    pass


class NeedResetCookieError(AuthentificationError):
    pass


class TwoCookiesRequiredError(AuthentificationError):
    pass


def get_real_ip(request):
    if 'HTTP_X_REAL_IP' in request:
        return request['HTTP_X_REAL_IP']
    if 'HTTP_X_FORWARDED_FOR' in request:
        return request['HTTP_X_FORWARDED_FOR'].split(',')[0]
    return request.get('REMOTE_ADDR', None)

def _get_session(cookie, userip, http_host):
        '''
        Получает ответ о состоянии сессии через blackbox, используя cookie
        '''
        bb = blackbox.XmlBlackbox(url=BLACKBOX_URL)

        session = bb.sessionid(cookie, userip, http_host)

        session.url = bb.url

        return session


def _get_sessionid(request, session_cookie_name=SESSIONID_COOKIE_NAME):
    try:
        cookie = SimpleCookie(request['HTTP_COOKIE']).get(session_cookie_name)
    except KeyError:
        return None

    return cookie.value if cookie else None


def check_authentification(request):
    cookie = _get_sessionid(request)

    if not cookie:
        logger.warning('user was not authentificated: no secret cookie')
        raise NotAuthentificatedError()

    userip = get_real_ip(request)
    http_host = request['HTTP_HOST']

    try:
        session = _get_session(cookie, userip, http_host)

        if session.valid:
            if session.redirect:
                # кука валидна, но требует продления по особой ссылке
                raise NeedResetCookieError()
            return True

        else:
            logger.warning('user was not authentificated: Session_id cookie is invalid: %s', session.error)
            raise NotAuthentificatedError()

    except blackbox.BlackboxError as exc:  # если blackbox недоступен, то пользователь анонимен
        logger.exception('Blackbox exception: %s', exc)
        raise


def login_required(func):
    """
    Декоратор, требует аутентифицикацию у пользователя
    """
    def wrapper(request, start_resp_callback, *args, **kwargs):
        if request['HTTP_HOST'] == 'h.yandex.net' and \
           request.get('REQUEST_URI', '') not in NO_AUTH_HANDLES:
            # В продакшен Хайдреферера некоторые все еще ходят через h.yandex.net,
            # редиректим таких товарищей на h.yandex-team.ru
            dest_url, lang = get_destination_url(request)
            yt_url = 'https://h.yandex-team.ru' + request.get('REQUEST_URI', '')
            logger.info('Request to h.yandex.net from %s', request.get('HTTP_REFERER', 'unknown referer'))
            return Redirect302Response().start(yt_url, language=lang, callback=start_resp_callback)

        # аутентификация не работает для системных ручек и фавиконки
        if request.get('REQUEST_URI', '') not in NO_AUTH_HANDLES and \
           request.get('QUERY_STRING', '') not in WHITE_LIST:

            logger.debug('checking user auth for cookie %s', request.get('HTTP_COOKIE', ''))
            try:

                check_authentification(request)

            except AuthentificationError as exc:
                dest_url, lang = get_destination_url(request)

                # если кука невалидна или нет sessionid2, отправляем пользователя в домик,
                # если валидна, но устарела -  отправляем на продление
                # документация:
                # https://beta.wiki.yandex-team.ru/passport/mda/intro/#povedenieservisovvkornevomdomeneidalnemzarubezhe
                passport_redir_url = COOKIE_RENEW_URL if isinstance(exc, NeedResetCookieError) else LOGIN_URL

                if passport_redir_url in request.get('HTTP_REFERER', ''):
                    # чтобы не попадать в redirect loop, кидаем ошибку
                    return InternalErrorResponse(exc).start(dest_url, language=lang, callback=start_resp_callback)
                else:
                    # конструируем url, на который паспорт вернет нас после авторизации
                    password_auth_url = ''.join([
                        passport_redir_url,
                        'retpath=',
                        quote('https://' + request['HTTP_HOST'] + request['REQUEST_URI'])
                    ])
                    logger.warning('redirecting to %s', password_auth_url)
                    return Redirect302Response().start(password_auth_url, language=lang, callback=start_resp_callback)

            except Exception as exc:
                # кидаем общую ошибку
                dest_url, lang = get_destination_url(request)
                logger.exception(exc)
                return InternalErrorResponse(exc).start(dest_url, language=lang, callback=start_resp_callback)

        return func(request, start_resp_callback, *args, **kwargs)

    return wrapper
