# coding: utf-8

from urllib.error import URLError
from urllib.parse import urlparse
from datetime import timedelta, datetime

import memcache
from .logger import logger

from .settings import MEMCACHED_HOST, DEBUG, CACHE, \
    GOOD_CACHE_STATUS_TIMEOUT, BAD_CACHE_STATUS_TIMEOUT

mc = memcache.Client([MEMCACHED_HOST], debug=1 if DEBUG else 0)

CID_DATE = 0
CID_RESULT = 1
CID_CONTENT = 2

URL_PREFIX = "hr_"


class WasInCacheException(Exception):
    """
    Exception, используемый для упрощения control flow, кидается в случае, если ответ был взят из кэша
    """
    pass


def get_key(url):
    return URL_PREFIX + url[:240]


def get_cached_status(purl):
    """
    Return url status from cache
    If redirect with META, return True, if do nothing return False
    http://wiki.yandex-team.ru/intranet/hidereferer/keshirovanie
    """

    chopped_purl = purl._replace(fragment="")
    key = get_key(chopped_purl.geturl().strip())
    status = mc.get(key)

    if status is None:
        return False

    # FIXME: remove this hack after fixing WIKI-4834 bug
    if len(status) != 3 or type(status[1]) != bool:
        logger.error('Memcache returned for key %s wrong data %s',
                     key, repr(status))
        # we received wrong cached data set by viruses.main.update_list function
        return False

    if status[CID_DATE] + timedelta(**CACHE['timeout']) < datetime.now():
        # cache expired
        return False
    elif status[CID_CONTENT].startswith('image/'):
        # will not cache images
        return False
    else:
        return status[CID_RESULT]


def cache_status(purl, result, content_type):
    """
    Save url status in cache
    @param purl:
    @param result:
    @param content_type:
    """
    chopped_purl = purl._replace(fragment="")
    mc.set(
        get_key(chopped_purl.geturl().strip()),
        (datetime.now(), result is True, content_type or ''),
        GOOD_CACHE_STATUS_TIMEOUT if result else BAD_CACHE_STATUS_TIMEOUT
    )


def cached(func):
    """
    cache decorator
    """

    def wrapper(dest_url, *args, **kwargs):
        """
        trying to get response from cache or set received response to cache
        """

        purl = urlparse(dest_url)

        # trying to get response from cache
        if get_cached_status(purl):
            raise WasInCacheException

        # original response func
        response = func(dest_url, *args, **kwargs)

        content_type = response.headers.get('content-type', '')
        # will not cache images
        if content_type.startswith('image/'):
            cache_status(purl, False, content_type)
        else:
            cache_status(purl, True, content_type)

        return response

    return wrapper
