# -*- coding: utf-8 -*-

import urllib.request, urllib.error, urllib.parse
from http.client import InvalidURL
from urllib.parse import urlparse, urlunparse
from encodings import idna
from .cache import cached


def get_headers(request):
    """
    Get headers from request
    """

    headers = {}
    for key in list(request.keys()):
        if key.startswith('HTTP_') and\
            not key.startswith('HTTP_X_') and\
                isinstance(request[key], str):

            header_name = '-'.join([part.lower().capitalize() for part in key[5:].split('_')])
            headers[header_name] = request[key]

    headers_to_remove = ('Cookie', 'Host', 'Referer')
    for header in headers_to_remove:
        if header in headers:
            del headers[header]

    return headers


@cached
def get_external_url(url, request):
    '''
    Returns somewhat an iterator content behind the URL and additional info

    @param parsed_url: url parsed by URL parse
    '''

    try:
        url = convert_idna(url)
    except UnicodeError:
        raise InvalidURL

    headers = get_headers(request)
    remote_request = urllib.request.Request(url,
                                     headers=headers,
                                     unverifiable=True)
    return urllib.request.urlopen(remote_request,
                           timeout=1)


def convert_idna(url):
    purl = urlparse(url)
    if any(ord(c) < 128 for c in purl.netloc):
        #convert it
        try:
            domain, port = purl.netloc.split(':')
        except ValueError:
            domain = purl.netloc
            port = None

        if not isinstance(domain, str):
            domain = domain.decode('utf-8')

        domain = '.'.join([idna.ToASCII(domain).decode('utf-8') for domain in domain.split('.')])
        if port is not None:
            domain += port
        return urlunparse((purl.scheme, domain, purl.path, purl.params, purl.query, purl.fragment))
    return purl.geturl()


class HTTPBasicAuthHandler(urllib.request.HTTPBasicAuthHandler):
    tries = 0
    maxtries = 1

    def http_error_401(self, req, fp, code, msg, headers):
        self.tries += 1
        url = req.get_full_url()

        if self.tries > 5:
            raise urllib.error.HTTPError(req.get_full_url(), 401, "basic auth failed",
                                    headers, None)
        else:
            self.tries += 1

        return self.http_error_auth_reqed('www-authenticate',
                                          url, req, headers)
