# coding: utf-8
import http.client
import traceback

from html import escape
from urllib.parse import quote

from . import settings
from .logger import logger
from .messages import *


class BaseResponse(object):
    """
    base response object class
    """

    _code = '403 Forbidden'
    _headers = {'Content-Type': 'text/html; charset=UTF-8'}
    _body = DEFAULT_MESSAGE

    _log_msg = 'Forbidden url: %s'
    _lang = 'en'

    def __init__(self, *args, **kwargs):
        super(BaseResponse, self).__init__()

        self.code = kwargs.get('code', self._code)
        self.headers = kwargs.get('headers', self._headers)
        self.body = kwargs.get('body', self._body)

    def get_body_string(self, *args, **kwargs):
        """
        Get body string from self.body temlate
        """
        body_string = self.body
        if isinstance(self.body, dict):
            # body text will be in user preferred language or in default _lang
            body_string = self.body[self.language]

        if isinstance(body_string, str):
            body_string = body_string.encode('utf-8')
        return body_string

    def start(self, dest_url, language=None, callback=None):
        """
        Start response sequence and return response body

        @param dest_url: redirect destination url
        @param language: response language
        @param callback: response sequence start hook -- for setting response
                            code and headers
        """
        self.language = (language or self._lang)

        log_args = list(filter(bool, (self._log_msg, dest_url)))

        logger.debug(*log_args)

        if callback and callable(callback):
            # prefill response code and headers
            callback(self.code, list(self.headers.items()))

        return self.get_body_string(dest_url)

    def __repr__(self):
        return '%s %s\n %s\n %s' % (self.__class__, self.code,
                                    self.headers, self.body)


class NoUrlResponse(BaseResponse):
    """
    Answer that no url is given to hiderefferer
    """
    _code = '200 OK'
    _body = NO_URL

    _log_msg = 'No url received'


class MetaRedirectResponse(BaseResponse):
    """
    Answer with code 200 and meta redirect in response body
    """
    _code = '200 OK'
    _body = TEMPLATE

    _log_msg = 'Redirect via meta to "%s"'

    def get_body_string(self, dest_url):
        """
        Get body string from self.body temlate
        """
        return self.body % {'url_anchor': dest_url}


class Redirect302Response(BaseResponse):
    """
    Answer with 302 redirect
    """

    _code = '302 Found'
    _body = ''

    _log_msg = 'Redirecting client to "%s"'

    def start(self, dest_url, language=None, callback=None):
        """
        Start response sequence and return response body
        """
        self.headers = {'Location': dest_url}

        return super(Redirect302Response, self)\
            .start(dest_url, language, callback)


class InfectedResponse(BaseResponse):
    """
    Answer with code 302 redirect to caution page in case of infected link
    """
    _code = '302 Found'
    _body = ''

    _log_msg = 'Infected link "%s", redirecting client to caution page'

    def start(self, dest_url, language=None, callback=None):
        """
        Start response sequence and return response body
        """
        caution_page_link = settings.INFECTED_LINK_CAUTION_PAGE.format(
            link=quote(dest_url, safe='')
        )
        self.headers = {'Location': caution_page_link}

        return super(InfectedResponse, self)\
            .start(dest_url, language, callback)


class ImageResponse(BaseResponse):
    """
    Answer with image
    """
    _log_msg = 'Processing image at URL: "%s"'

    def __init__(self, resource, *args, **kwargs):
        super(ImageResponse, self).__init__(*args, **kwargs)

        self.code = '%s %s' % (resource.code, http.client.responses[resource.code])
        self.headers = dict(resource.headers)

        self.headers['Accept-Ranges'] = 'none'
        if 'set-cookie' in self.headers:
            del self.headers['set-cookie']

        # TOOLSUP-5794: В случае transfer-encoding: chunked браузеры выдают
        # ошибку при загрузке картики (net::ERR_EMPTY_RESPONSE в случае Chromium),
        # т.к. они пытаются интерпретировать возвращаемые данные как chunk,
        # при этом urllib2 за один read() выкачивает все chunk'и.
        if 'transfer-encoding' in self.headers:
            del self.headers['transfer-encoding']

        self.body = resource


class NotModifiedResponse(BaseResponse):
    """
    Answer with code 304 Not modified
    """
    _log_msg = 'Resource "%s" is not modified'

    def __init__(self, resource, *args, **kwargs):
        super(NotModifiedResponse, self).__init__(*args, **kwargs)

        self.code = '%s %s' % (resource.code, resource.msg)
        self.headers = dict(resource.hdrs)

        if 'set-cookie' in self.headers:
            del self.headers['set-cookie']

        self.body = resource.fp


class InternalErrorResponse(BaseResponse):
    """
    Answer with server error 500
    """
    _code = '500 Internal Server Error'

    def __init__(self, exception='', *args, **kwargs):
        super(InternalErrorResponse, self).__init__(*args, **kwargs)

        self.exception = (exception or traceback.format_exc())

    def start(self, dest_url, language=None, callback=None):
        """
        Start response sequence and return response body
        """
        self.language = (language or self._lang)

        if callback and callable(callback):
            # prefill response code and headers
            callback(self.code, list(self.headers.items()))

        body_string = self.get_body_string(dest_url)
        if settings.DEBUG:
            body_string = self.exception

        return body_string

    def get_body_string(self, dest_url, *args, **kwargs):
        body_string = super(InternalErrorResponse, self).get_body_string(*args, **kwargs)
        return '{0} ({1})'.format(body_string.decode('utf-8'), escape(dest_url, quote=False))
