import dateutil.parser
import os
import requests

from datetime import date
from dateutil.relativedelta import relativedelta
from typing import Dict, Any

from ids.registry import registry
import startrek_client


st_host = 'st-api.yandex-team.ru'

services = {
    'staff': {
        'Дежурный по бекенду': {
            'queue': 'STAFF',
            'tags': ['duty-log'],
            'assignee': '{duty_login}',
            'summary': 'Дежурство @{duty_login} - {duty_from} - {duty_to}',
            'description': """Лог дежурства
((https://wiki.yandex-team.ru/staff/dev/duty/monitors/ как реагировать на мониторинги))

- Смотри ((https://st.yandex-team.ru/dashboard/37407 дашборд дежурного))
- Чини по ((https://wiki.yandex-team.ru/staff/dev/duty/ инструкции))
- ((https://wiki.yandex-team.ru/staff/dev/duty/fakap/ Если уж совсем все плохо))

((https://a.yandex-team.ru/arcadia/intranet/hr-tech-duty-robot/README.md как призвать дежурного))""",
        },

        'Дежурный по фронтенду': {
            'queue': 'STAFF',
            'tags': ['duty-log', 'front'],
            'assignee': '{duty_login}',
            'summary': 'Дежурство @{duty_login} - {duty_from} - {duty_to}',
            'description': """Лог дежурства
((https://st.yandex-team.ru/dashboard/62603 Дашборд дежурного))""",
        },
    }
}


def create_duty_log(issue_options: Dict[str, Any], duty_login: str, duty_from: str, duty_to: str, oauth_token: str) -> None:
    creation_options: Dict[str, Any] = dict()

    for key, value in issue_options.items():
        if isinstance(value, str):
            creation_options[key] = value.format(**locals())
        else:
            creation_options[key] = value

    creation_options['unique'] = creation_options['summary']

    repository = registry.get_repository(
        service='startrek2',
        resource_type='issues',
        oauth_token=oauth_token,
        user_agent='python',
        protocol='https',
        host=st_host,
        timeout=30,
    )

    if not repository.get(lookup={'filter': f'summary:{creation_options["summary"]}'}):
        repository.create(**creation_options)


oauth_token = os.getenv('ROBOT_DUTY_OAUTH_TOKEN')

if not oauth_token:
    with open('C:\\Users\\tikhonov-ka\\.ya_token', 'r') as file:
        oauth_token = file.read().rstrip()


date_from = date.today()
date_to = date_from + relativedelta(days=7)
fields = ['is_approved', 'person.login', 'schedule.name', 'start', 'end', 'replaces.person.login', 'replaces.schedule.name', 'replaces.start', 'replaces.end']

query = {
    'date_from': date_from.strftime('%Y-%m-%d'),
    'date_to': date_to.strftime('%Y-%m-%d'),
    'show_in_staff': True,
    'fields': ','.join(fields),
}

for service, duty_logs in services.items():
    query['service__slug'] = service
    abc_response = requests.get(
        url='https://abc-back.yandex-team.ru/api/v4/duty/shifts/',
        headers={'Authorization': f'OAuth {oauth_token}'},
        params=query,
        timeout=30,
    )

    for shift in abc_response.json().get('results', []):
        if not shift['is_approved']:
            continue

        schedule_name = shift['schedule']['name']
        issue_options = duty_logs.get(schedule_name, None)

        if issue_options is not None:
            duty_login = shift['person']['login']
            duty_from = shift['start']
            duty_to = shift['end']
            create_duty_log(
                issue_options,
                duty_login,
                dateutil.parser.parse(duty_from).strftime("%d.%m.%Y"),
                dateutil.parser.parse(duty_to).strftime("%d.%m.%Y"),
                oauth_token,
            )
