from django.http import JsonResponse
from django.core.exceptions import ObjectDoesNotExist
import json
import logging
import traceback

from intranet.hrdb_ext.src.amo.models import TicketRequest
from intranet.hrdb_ext.src.amo.core.create import process_ticket_request
from intranet.hrdb_ext.src.amo.serializers import TicketRequestSerializer
from intranet.hrdb_ext.src.amo.exceptions import NotFoundException, BadRequestException
from intranet.hrdb_ext.src.amo.core.sync import sync_request_ticket
from intranet.hrdb_ext.src.amo.tasks import push_tickets_states_to_amo
from intranet.hrdb_ext.src.lib.views import ExternalAuthenticationView, YandexAuthenticationView

logger = logging.getLogger(__name__)


def get_request(uuid=None, external_id=None, key=None):
    if uuid is not None:
        try:
            return TicketRequest.objects.get(uuid=uuid)
        except ObjectDoesNotExist:
            raise NotFoundException('Invalid uuid')
    elif external_id is not None:
        try:
            return TicketRequest.objects.get(external_id=external_id)
        except ObjectDoesNotExist:
            raise NotFoundException('Invalid external_id')
    elif key is not None:
        try:
            return TicketRequest.objects.get(ticket_key=key)
        except ObjectDoesNotExist:
            raise NotFoundException('Invalid key')

    raise BadRequestException('request key not specified')


def get_body(request):
    body = request.POST
    if not body:
        body = json.loads(request.body.decode('utf-8'))
    if isinstance(body, str):
        return json.loads(body)
    return body


class BaseAuthenticationView(ExternalAuthenticationView):
    def dispatch(self, request, *args, **kwargs):
        logger.info('Request to {}'.format(self.__class__.__name__))
        try:
            return super(BaseAuthenticationView, self).dispatch(request, *args, **kwargs)
        except NotFoundException as e:
            logger.error(traceback.format_exc())
            return JsonResponse({'error': str(e)}, status=404)
        except BadRequestException as e:
            logger.error(traceback.format_exc())
            return JsonResponse({'error': str(e)}, status=400)
        except Exception as e:
            logger.error(traceback.format_exc())
            return JsonResponse({'error': str(e)}, status=500)


class CreateIssueView(BaseAuthenticationView):
    def post(self, request):
        body = get_body(request)
        issue_request = process_ticket_request(
            body['body'],
            external_id=body.get('external_id', None),
        )
        serialized = TicketRequestSerializer(issue_request)
        return JsonResponse(serialized.data, status=201)


class GetIssueRequestView(BaseAuthenticationView):
    def get(self, request, uuid):
        issue_request = get_request(uuid=uuid)

        need_sync = request.GET.get('sync', None)
        if need_sync in [1, '1', True, 'true']:
            issue_request = sync_request_ticket(issue_request)

        serialized = TicketRequestSerializer(issue_request)
        return JsonResponse(serialized.data)


class TaskIssuesPushRunView(YandexAuthenticationView):
    def get(self, request):
        future = push_tickets_states_to_amo.delay()
        logger.info('{}: {}'.format(self.__class__.__name__, future))
        return JsonResponse({
            'status': 'ok',
            'message': 'Push task registered',
            'future': str(future),
        })
