"""
Inspired by IDS
https://a.yandex-team.ru/arcadia/library/python/ids
"""

import logging
import requests
import os
from urllib.parse import urlencode, quote_plus
from string import Formatter

from django.conf import settings
from intranet.hrdb_ext.src.ispring.exceptions import ConnectorException


logger = logging.getLogger(__name__)


def get_session():
    session = requests.Session()
    session.verify = os.path.exists(settings.INTERNAL_CA_CERTIFICATE_PATH)
    return session


class HttpConnector:
    resources = {
        'department': '/department/{department_id}',
        'departments_all': '/department',
        'departments_iter': '/departments',

        'user': '/user/{user_id}',
        'users_all': '/user',
        'users_iter': '/users',
    }

    def __init__(self, host=None, **kwargs):
        self.session = get_session()
        self.host = host or settings.ISPRING_API_HOST
        self.timeout = kwargs.pop('timeout', 2)
        self.retries = kwargs.pop('retries', 1)
        self.statuses_to_retry = kwargs.pop('statuses_to_retry', [500])

        self.headers = {
            'User-Agent': settings.ISPRING_USER_AGENT,
            'X-Auth-Account-Url': settings.ISPRING_ACCOUNT_ID,
            'X-Auth-Email': settings.ISPRING_LOGIN,
            'X-Auth-Password': settings.ISPRING_PASSWORD,
        }

    def execute_request(self, method='get', resource=None, url_vars=None, **params):
        params.setdefault('timeout', self.timeout)
        url = self.build_url(resource=resource, url_vars=url_vars)
        return self._try_request(method, url, **params)

    def build_url(self, resource=None, url_vars=None):
        url_vars = url_vars or {}

        resource_url = self.resources.get(resource, '')
        url = '{}{}'.format(self.host, resource_url.format(**url_vars))

        query_params = self._get_query_params(resource_url, url_vars)
        if query_params:
            url = '{}?{}'.format(url, urlencode(query_params, quote_via=quote_plus))

        return url

    def _try_request(self, method, url, **params):
        response = None
        for attempt in range(max(self.retries, 1)):
            try:
                logger.debug('{} Url={}. Params={}'.format(method.upper(), url, params))
                response = self.session.request(method.upper(), url, **params, headers=self.headers)
                if response.status_code in self.statuses_to_retry:
                    logger.warning(response.text)
                    continue
                break
            except Exception as e:
                logger.exception('HTTP error ({}): {}, "{}"'.format(attempt, url, e))

        if not response:
            raise ConnectorException('No success request')

        if response.status_code > 299:
            logger.warning(response.text)
            raise ConnectorException('Non-success response code: {}'.format(response.status_code))

        return response

    def _get_query_params(self, resource_url, url_vars):
        params = [i[1] for i in Formatter().parse(resource_url) if i[1] is not None]
        return {key: value for key, value in url_vars.items() if key not in params}

    def get(self, *args, **kwargs):
        return self.execute_request('GET', *args, **kwargs)

    def post(self, *args, **kwargs):
        return self.execute_request('POST', *args, **kwargs)

    def delete(self, *args, **kwargs):
        return self.execute_request('DELETE', *args, **kwargs)
