from xml.etree.ElementTree import fromstring
from copy import deepcopy


class Page:
    def __init__(self, response):
        self.response = response

    def __iter__(self):
        for item in self.extract_items():
            yield item

    def extract_items(self):
        raise NotImplementedError

    def next_page_token(self):
        raise NotImplementedError


class XmlPage(Page):
    item_key = None
    next_page_key = 'nextPageToken'

    def __init__(self, response):
        super(XmlPage, self).__init__(response)
        if response.status_code not in [200, 201]:
            raise Exception('Trying to parse non-success request')
        self.root = fromstring(response.content.decode('utf-8'))

    def extract_items(self):
        return self.root.findall('.//{}'.format(self.item_key))

    def next_page_token(self):
        token = self.root.find('.//nextPageToken')
        return token.text if token is not None else None


class Fetcher:
    def __init__(self, connector, page_cls, **request_params):
        self.connector = connector
        self.page_cls = page_cls or XmlPage
        self.request_params = request_params

    def get_page(self, previous=None):
        params = deepcopy(self.request_params)
        if previous and previous.next_page_token():
            url_vars = params.get('url_vars', {})
            url_vars['pageToken'] = previous.next_page_token()
            params['url_vars'] = url_vars
        return self.page_cls(self.connector.get(**params))


class ResultSet:
    def __init__(self, fetcher, wrapper):
        self.fetcher = fetcher
        self.wrapper = wrapper

    def __iter__(self):
        for page in self.get_pages():
            for obj in page:
                yield self.wrapper.from_xml(obj)

    def get_pages(self):
        first_page = self.fetcher.get_page()
        yield first_page

        current_page = first_page
        while current_page.next_page_token() is not None:
            next_page = self.fetcher.get_page(previous=current_page)
            yield next_page
            current_page = next_page
