from intranet.hrdb_ext.src.ispring.mock.response import Code, MockedResponse
from xml.etree.ElementTree import Element


class Bucket:
    def __init__(self, object_cls, obj_id_key):
        self.object_cls = object_cls
        self.obj_id_key = obj_id_key
        self.by_id = {}
        self.paginator = None

    def generate(self, count=5):
        for _ in range(count):
            obj = self.object_cls.generate()
            self.by_id[obj.uid()] = obj
        return list(self.by_id.values())

    def filter_objects(self, url_vars):
        return [value for key, value in sorted(self.by_id.items())]

    def handle(self, method, resource, url_vars, **params):
        obj_id = url_vars.get(self.obj_id_key, None)
        if method == 'get':
            if resource.endswith('_iter'):
                return self.get_pages(url_vars, **params)
            elif resource.endswith('_all'):
                return self.get_all(url_vars, **params)
            return self.get_one(obj_id)
        elif method == 'post':
            if resource.endswith('_all'):
                return self.create(params['json'])
            return self.update(obj_id, params['json'])
        elif method == 'delete':
            return self.delete(obj_id)
        raise ValueError('Unknown method {}'.format(method))

    def get_pages(self, url_vars, **params):
        page_size = url_vars.get('pageSize', 2)
        page_token = url_vars.get('pageToken', None)

        if (self.paginator is None) or (page_token is None):
            objects = self.filter_objects(url_vars)
            self.paginator = Paginator(objects, page_size=page_size)

        page, next_page_token = self.paginator.next(page_token)
        root = self.paginator.serialize(page, next_page_token)
        return MockedResponse(root)

    def get_all(self, url_vars, **params):
        url_vars = url_vars or {}
        url_vars['pageSize'] = 1000000000
        return self.get_pages(url_vars, **params)

    def get_one(self, user_id):
        obj = self.by_id.get(user_id, None)
        if not obj:
            return MockedResponse(None, code=Code.NOT_FOUND)
        root = Element('response')
        root.append(obj.serialize())
        return MockedResponse(root)

    def create(self, fields):
        obj = self.object_cls.create(**fields)
        root = Element('response')
        root.text = obj.uid()
        self.by_id[obj.uid()] = obj
        return MockedResponse(root)

    def update(self, obj_id, fields):
        obj = self.by_id.get(obj_id, None)
        if obj is None:
            return MockedResponse(None, code=Code.NOT_FOUND)
        obj.update(fields)
        return MockedResponse(None)

    def delete(self, obj_id):
        if obj_id not in self.by_id:
            return MockedResponse(None, code=Code.NOT_FOUND)
        self.by_id.pop(obj_id, None)
        return MockedResponse(None)


class Paginator:
    def __init__(self, objects, page_size):
        self.objects = objects
        self.page_size = page_size
        self.total_pages = (len(objects) + page_size - 1) // page_size

    def next(self, page_token=None):
        page = int(page_token or 0)
        next_page_token = page + 1 if page + 1 < self.total_pages else None
        return self.get_page(page), next_page_token

    def get_page(self, page):
        return self.objects[page * self.page_size:(page + 1) * self.page_size]

    def serialize(self, objects, next_page_token):
        root = Element('response')

        bucket = Element(objects[0].response_key_list)
        for obj in objects:
            bucket.append(obj.serialize())
        root.append(bucket)

        if next_page_token:
            node = Element('nextPageToken')
            node.text = str(next_page_token)
            root.append(node)

        return root
