import attr
import uuid

from xml.etree.ElementTree import fromstring
from intranet.hrdb_ext.src.ispring.api.departments import ISpringDepartment
from intranet.hrdb_ext.src.ispring.api.users import ISpringUser


class AbstractResourceMixin:
    response_key = None
    response_key_list = None

    def uid(self):
        raise NotImplementedError

    @classmethod
    def create(cls, **fields):
        raise NotImplementedError

    @classmethod
    def generate(cls):
        return cls(**{
            attribute.name: str(uuid.uuid4())
            for attribute in cls.__attrs_attrs__
        })

    def update(self, fields):
        for field, value in fields.items():
            if hasattr(self, field) and field:
                setattr(self, field, value)

    def to_xml_node(self, string):
        return fromstring(string)


@attr.s
class MockedUser(ISpringUser, AbstractResourceMixin):
    response_key = 'userProfile'
    response_key_list = 'userProfiles'

    template = """
    <userProfile>
        <userId>{user_id}</userId>
        <departmentId>{department_id}</departmentId>
        <fields>
            <field>
                <name>FIRST_NAME</name>
                <value>{first_name}</value>
            </field>
            <field>
                <name>LAST_NAME</name>
                <value>{last_name}</value>
            </field>
            <field>
                <name>LOGIN</name>
                <value>{login}</value>
            </field>
        </fields>
    </userProfile>
    """

    def uid(self):
        return self.user_id

    @classmethod
    def create(cls, **fields):
        return cls(
            user_id=str(uuid.uuid4()),
            department_id=fields.get('department_id', fields.get('departmentId', None)),
            login=fields.get('login', None),
            first_name=fields.get('first_name', None),
            last_name=fields.get('last_name', None),
        )

    def serialize(self):
        return self.to_xml_node(self.template.format(
            user_id=self.user_id,
            login=self.login,
            department_id=self.department_id,
            first_name=self.first_name,
            last_name=self.last_name,
        ))


@attr.s
class MockedDepartment(ISpringDepartment, AbstractResourceMixin):
    response_key = 'department'
    response_key_list = 'departments'

    template = """
    <department>
        <departmentId>{department_id}</departmentId>
        <parentDepartmentId>{parent_id}</parentDepartmentId>
        <name>{name}</name>
        <code>{code}</code>
    </department>
    """

    def uid(self):
        return self.department_id

    @classmethod
    def create(cls, **fields):
        return cls(
            department_id=str(uuid.uuid4()),
            name=fields.get('name', None),
            code=fields.get('code', None),
            parent_id=fields.get('parent_id', fields.get('parentDepartmentId', None)),
        )

    def serialize(self):
        return self.to_xml_node(self.template.format(
            department_id=self.department_id,
            parent_id=self.parent_id,
            name=self.name,
            code=self.code,
        ))
