import attr
from django.conf import settings
import logging
import traceback

from intranet.hrdb_ext.src.ispring.sync.task import AbstractSyncTask
from intranet.hrdb_ext.src.ispring.api.pagination import XmlPage
from intranet.hrdb_ext.src.ispring.api.departments import DepartmentRepository
from intranet.hrdb_ext.src.ispring.sync.tree import load_staff_tree, load_ispring_tree


logger = logging.getLogger(__name__)


class AbstractDepartmentTask(AbstractSyncTask):
    """
    Abstract task for synchronizing ISpring and Staff departments trees.
    See also: intranet.hrdb_ext.src.ispring.sync.tree.py
    """

    # Controls behaviour of self.run_requests().
    # If set to True: we exit after first failed request.
    # If set to False: do not stop after fail (just logging the error)
    FAIL_INSTANTLY = False

    def __init__(self, ispring_root_dep_id, staff_dep_id, repository=None):
        """
        :param ispring_root_dep_id:
            ISpring department, which should be parent of staff department.
        :param staff_dep_id:
            Staff root department ID, which we want to sync.
        :param repository:
            DepartmentRepository instance
        """
        super(AbstractDepartmentTask, self).__init__()
        self.ispring_root_dep_id = ispring_root_dep_id
        self.staff_dep_id = staff_dep_id

        self.repository = repository or DepartmentRepository()
        self.ispring_tree = None
        self.staff_tree = None

    def load(self, include_deleted=False):
        self.ispring_tree = load_ispring_tree(repository=self.repository)
        self.staff_tree = load_staff_tree(include_deleted=include_deleted)

    def log(self, message, level=logging.INFO):
        message = '[{}] {}'.format(self.__class__.__name__, message)
        logger.log(level, message)

    def run_requests(self):
        count = len(self.requests)
        self.log('Requests to run: {}'.format(count))

        for i, request in enumerate(self.requests):
            self.log('Running request {}/{}: {}'.format(i, count, request))
            try:
                self.execute_request(request)
            except Exception as e:
                self.log(str(e), level=logging.ERROR)
                self.log(traceback.format_exc(), level=logging.ERROR)
                if self.FAIL_INSTANTLY:
                    raise e

    def execute_request(self, request):
        lookup = {'department_id': request.department_id}
        response = self.repository.update(lookup, body=request.to_json())
        if response.status_code not in [200, 201, 204]:
            raise ValueError(f'Failed to update "{lookup}": {response.status_code}')


@attr.s
class DepartmentCreateRequest:
    department_id = attr.ib()
    name = attr.ib()
    code = attr.ib()

    parent_exists = attr.ib()

    # May contain two different values:
    # 1. Existing ISpring department ID
    # 2. Staff department url. That means that parent was created just now.
    parent_id = attr.ib()

    def to_json(self):
        return {
            'parentDepartmentId': self.parent_id,
            'code': self.code,
            'name': self.name,
        }


class CreateDepartments(AbstractDepartmentTask):
    """
    Task which creates new nodes in ISpring departments tree.

    * Using BFS to traverse the tree, because we need active parent before creating a child.
    * Looking only into specified staff branch (by root department id).
    * FAIL_INSTANTLY = True - for the same reason why we use BFS.
    """

    FAIL_INSTANTLY = True

    def __init__(self, *args, **kwargs):
        super(CreateDepartments, self).__init__(*args, **kwargs)
        self.new_departments = {}

    def generate_requests(self):
        queue = [(
            self.staff_dep_id,          # Current staff department ID
            self.ispring_root_dep_id,   # ISpring PARENT department ID for current staff dep ID
            0,                          # Current depth
            True,                       # Is parent if current staff department ID exists in ISpring
        )]

        while queue:
            department_id, ispring_parent_id, depth, parent_exists = queue.pop(0)

            staff_department = self.staff_tree.by_id[department_id]
            ispring_department = self.ispring_tree.by_code.get(staff_department.code, None)

            exists = False
            ispring_id = staff_department.code
            if ispring_department is not None:
                exists = True
                ispring_id = ispring_department.department_id

            if not exists:
                request = DepartmentCreateRequest(
                    department_id=None,
                    parent_id=ispring_parent_id,
                    name=staff_department.name,
                    code=staff_department.code,
                    parent_exists=parent_exists,
                )

                self.log(f'Add request "{ispring_parent_id}"->"{department_id}"')
                self.requests.append(request)

            for child in self.staff_tree.children.get(department_id, []):
                queue.append((child, ispring_id, depth + 1, exists))

    def get_parent_id(self, request):
        """
        Recognizing ISpring parent department ID for the department,
        which will be created by specified request.

        There are 2 possible scenarios:
        1. Parent already exists in ISpring (see request.parent_exists).
        2. We just created a parent. Extracting its ID from self.new_departments.

        See also: doc for request.parent_id.
        """
        if request.parent_exists:
            return request.parent_id

        parent_code = request.parent_id

        parent_from_new = self.new_departments.get(parent_code, None)
        if parent_from_new:
            return parent_from_new

        parent_from_known = self.ispring_tree.by_code.get(parent_code, None)
        if parent_from_known:
            return parent_from_known.department_id

        raise ValueError('Unknown parentDepartmentId for request {}'.format(request))

    def execute_request(self, request):
        """
        Creating department in ISpring.

        Saving new ISpring department ID in self.new_departments.
        Doing so to be able to create child departments with corrent parent.
        """
        request.parent_id = self.get_parent_id(request)
        response = self.repository.create({}, body=request.to_json())

        object_id = XmlPage(response).root.text
        self.new_departments[request.code] = object_id
        logger.info('Created new department {} for request {}'.format(object_id, request))


@attr.s
class DepartmentUpdateRequest:
    department_id = attr.ib()
    parent_id = attr.ib(default=None)
    name = attr.ib(default=None)

    json_fields = {
        'parent_id': 'parentDepartmentId',
        'name': 'name',
    }

    def anything_changed(self):
        return any(getattr(self, field) for field in self.json_fields)

    def to_json(self):
        body = {}
        for field, key in self.json_fields.items():
            value = getattr(self, field)
            if value:
                body[key] = value
        return body


class UpdateDepartments(AbstractDepartmentTask):
    """
    Task to update information about departments, which already exist in ISpring
    (and which are active on staff)

    Updating only:
    * parent_id
    * name

    Notes:
    * If we see that parent has changed, and it is unknown to us in ISpring -
        change parent to some "Trash department".
        Need separate task to clean up this "Trash"
    * We are assuming here, that all departments, which could be created, already has been created.
        (i.e. CreateDepartments task was executed before)
    """

    # It is OK to update only part of departments.
    # Task is idempotent: we can execute it again safely.
    FAIL_INSTANTLY = False

    def generate_requests(self):
        for node in self.staff_tree.bfs(self.staff_dep_id):
            department_id, parent_id, depth = node
            staff_department = self.staff_tree.by_id[department_id]

            code = staff_department.code
            ispring_department = self.ispring_tree.by_code.get(code, None)
            if ispring_department is None:
                self.log('Department {} not found in ISpring'.format(staff_department.code))
                continue

            request = self.generate_request(staff_department, ispring_department)
            if request is None:
                continue

            self.log(f'Add request "{ispring_department.department_id} ({department_id})"')
            self.requests.append(request)

    def generate_request(self, staff_department, ispring_department):
        """
        Checking if department has changed. If not - do nothing.
        If parent has changed, and we don't have it in ISpring - moving department to "Trash".
        """
        parent_id = self.get_parent_id(staff_department)
        if parent_id is None:
            parent_id = settings.ISPRING_TRASH_DEPARTMENT_ID
            self.log(
                'Parent for "code={}" not found in ispring. '
                'Using trash dep "{}"'.format(staff_department.code, parent_id)
            )

        request = DepartmentUpdateRequest(department_id=ispring_department.department_id)

        if parent_id != ispring_department.parent_id:
            request.parent_id = parent_id
        if staff_department.name != ispring_department.name:
            request.name = staff_department.name

        if not request.anything_changed():
            return None

        return request

    def get_parent_id(self, staff_department):
        staff_parent = self.staff_tree.get(staff_department.parent_id)
        if staff_parent is None:
            return None

        ispring_parent = self.ispring_tree.by_code.get(staff_parent.code, None)
        if ispring_parent is None:
            return None

        return ispring_parent.department_id

    def execute_request(self, request):
        lookup = {'department_id': request.department_id}
        response = self.repository.update(lookup, body=request.to_json())
        if response.status_code not in [200, 201, 204]:
            raise ValueError(f'Failed to update "{lookup}": {response.status_code}')


class RemoveUnknownDepartments(AbstractDepartmentTask):
    """
    Task to remove old departments from ISpring.

    CRUCIAL NOTES:
    * ISpring tree may contain departments, which were not created by us.
        We need to skip such departments.
        Removing only those, which were at some time under current staff department on staff.
    * You can not delete department if there are any user attached to it.
        So this task should be executed only after users list has been updated.
    * You can not delete department if it has child departments.
        So this task should be executed in DFS order (or reversed BFS order).
    """

    # It is OK to delete only part of departments.
    # Task is idempotent: we can continue from where we left of.
    FAIL_INSTANTLY = False

    def load(self, include_deleted=False):
        return super(RemoveUnknownDepartments, self).load(include_deleted=True)

    def generate_requests(self):
        staff_department = self.staff_tree.by_id.get(self.staff_dep_id, None)
        if staff_department is None:
            self.log('Specified root "{}" not found on staff. Exit.'.format(self.staff_dep_id))
            return

        ispring_department = self.ispring_tree.by_code.get(staff_department.code, None)
        if ispring_department is None:
            self.log('Not found mirror ispring department for {}. Exit'.format(self.staff_dep_id))
            return

        departments = [
            node[0]
            for node in self.ispring_tree.bfs(ispring_department.department_id)
        ]

        for department_id in departments[::-1]:
            ispring_department = self.ispring_tree.by_id[department_id]

            staff_department = self.staff_tree.by_code.get(ispring_department.code, None)
            if staff_department is None:
                self.log(
                    f'ISpring department "{department_id}" not found in staff. '
                    f'Skip from removal (not ours)'
                )
                continue

            if staff_department.is_deleted:
                self.log(
                    f'Department {department_id} ({ispring_department.code}) '
                    f'is deleted on staff. Add request to delete "{department_id}"',
                )
                self.requests.append(department_id)

    def execute_request(self, request):
        self.repository.delete({'department_id': request})
