import logging

from intranet.hrdb_ext.src.ispring.api.departments import ISpringDepartment
from intranet.hrdb_ext.src.ispring.api.departments import DepartmentRepository

from intranet.hrdb_ext.src.lib.staff import load_departments


logger = logging.getLogger(__name__)


class Tree:
    def __init__(self, departments):
        self.departments = departments

        self.by_id = {d.department_id: d for d in self.departments}
        self.by_code = {d.code: d for d in self.departments}

        self.children = {}
        for department in departments:
            if department.parent_id not in self.children:
                self.children[department.parent_id] = []
            self.children[department.parent_id].append(department.department_id)

    def get(self, key):
        if isinstance(key, ISpringDepartment):
            uid = key.department_id
            code = key.code
            return self.by_id.get(uid, self.by_code.get(code, None))
        return self.by_id.get(key, self.by_code.get(key, None))

    def bfs(self, root_id):
        queue = [(
            root_id,    # Текущий департамент
            None,       # Родитель
            0,          # Глубина
        )]
        while queue:
            department_id, parent_id, depth = queue.pop(0)
            yield department_id, parent_id, depth
            for child_id in self.children.get(department_id, []):
                queue.append((child_id, department_id, depth + 1))


def load_ispring_tree(repository=None):
    logger.debug('Loading ISpring tree')
    if repository is None:
        repository = DepartmentRepository()

    departments = repository.get_all({})
    logger.info('Loaded {} ISpring departments'.format(len(departments)))

    return Tree(departments)


def load_staff_tree(include_deleted=False):
    departments = load_departments(include_deleted=include_deleted)
    logger.info('Loaded {} staff departments'.format(len(departments)))

    parsed = list(map(ISpringDepartment.from_staff, departments))
    return Tree(parsed)
