from functools import wraps
import ylock
from django.conf import settings
import hashlib
import logging
from celery.exceptions import Ignore

logger = logging.getLogger(__name__)

lock_manager = ylock.create_manager(**settings.YLOCK)


def get_lock_name(function, args, kwargs):
    lock_name = f'{function.__module__}.{function.__name__}'
    if args or kwargs:
        hashed_args = hashlib.sha1(f'{args}-{kwargs}'.encode('utf-8')).hexdigest()
        lock_name += f':{hashed_args}'
    return lock_name


def locked_task(task=None, *, timeout=10, block=False, delay=10):
    assert task is None or callable(task)
    if callable(task):
        return locked_task()(task)

    def decorator(function):
        @wraps(function)
        def wrapper(*args, **kwargs):
            if not settings.IS_CELERY_INSTANCE:
                return function(*args, **kwargs)

            lock_name = get_lock_name(function, args, kwargs)
            logger.info('Acquiring lock `%s`', lock_name)
            with lock_manager.lock(lock_name, timeout, block, delay=delay) as acquired:
                if acquired:
                    logger.info('Lock `%s` acquired', lock_name)
                    return function(*args, **kwargs)
                else:
                    logger.info('Lock `%s` NOT acquired', lock_name)
                    raise Ignore(f'Lock `{lock_name}` NOT acquired')
        return wrapper
    return decorator
