# coding: utf-8

from django.utils.encoding import force_text, force_bytes

from tastypie import http


class RestApiError(Exception):
    """
    Базовая ошибка. Не должна выбрасываться сама по себе, вместо этого нужно использовать наследников.
    У наследников должен быть определен уникальный error_code
    """
    error_code: str
    response_class = http.HttpConflict  # 409
    message = ''
    errors = None

    def __init__(self, errors=None, message=None, **extra):
        if message is not None:
            self.message = message
        if isinstance(errors, dict):
            self.errors = {key: errors if isinstance(errors, dict) else [force_text(error) for error in errors]
                           for key, errors in errors.items()}
            if message is None:
                common_errors = self.errors.get('__all__')
                if common_errors is not None:
                    self.message = '\n'.join(common_errors)
        elif isinstance(errors, str):
            self.message = errors
        self.extra = extra
        assert hasattr(self, 'error_code'), 'Class %s must have "error_code" attribute' % type(self)

    def __str__(self):
        return force_text(self.message)

    def __repr__(self):
        return self.message


class CannotWriteInReadOnlyState(RestApiError):
    response_class = http.HttpApplicationError  # 500
    error_code = 'READONLY_STATE'
    message = 'IDM is in a read-only state, cannot handle write intending request'


class InternalAPIError(RestApiError):
    response_class = http.HttpApplicationError  # 500
    error_code = 'INTERNAL_API_ERROR'
    message = 'An API that IDM uses returned an invalid response'


class UnhandledException(RestApiError):
    response_class = http.HttpApplicationError  # 500
    error_code = 'UNKNOWN_ERROR'
    message = 'Unhandled exception'


class ResourceAlreadyExists(RestApiError):
    """
    Ресурс уже существует.
    """
    error_code = 'ALREADY_EXISTS'
    message = 'Resource already exists'


class BadRequest(RestApiError):
    response_class = http.HttpBadRequest  # 400
    error_code = 'BAD_REQUEST'
    message = 'Invalid data sent'


class Conflict(RestApiError):
    response_class = http.HttpConflict
    error_code = 'CONFLICT'
    message = 'Request could not be processed because of conflict in the request'


class Forbidden(RestApiError):
    response_class = http.HttpForbidden  # 403
    error_code = 'FORBIDDEN'
    message = 'You don\'t have permission to access this resource'


class Unauthorized(RestApiError):
    response_class = http.HttpUnauthorized  # 401
    error_code = 'UNAUTHORIZED'
    message = 'User is not authenticated'


class NotFound(RestApiError):
    response_class = http.HttpNotFound
    error_code = 'NOT_FOUND'
    message = 'Object could not be found'
