# coding: utf-8


import logging

from django.db.models import Count, Q
from django.utils.translation import ugettext as _

from idm.api.exceptions import BadRequest
from idm.api.frontend import apifields
from idm.api.frontend import forms
from idm.api.frontend.base import FrontendApiResource
from idm.api.frontend.utils import fix_user_field_filters
from idm.core.models import Action, RoleNode
from idm.inconsistencies.models import Inconsistency
from idm.reports.tasks import make_report
from idm.utils.human import inflect
from tastypie import fields

log = logging.getLogger(__name__)


class ActionResource(FrontendApiResource):
    """
    Ресурс объектов истории действий над ролями
    """
    # Внешние ключи
    group = apifields.GroupForeignKey()
    system = apifields.SystemForeignKey()
    user = apifields.UserForeignKey()

    # Внешние ключи для detail вида
    approverequest = apifields.ApproveRequestForeignKey(full_list=False)
    impersonator = apifields.UserForeignKey(attribute='impersonator')
    inconsistency = apifields.InconsistencyForeignKey(full_list=False)
    membership = apifields.MembershipForeignKey(full_list=False)
    requester = apifields.UserForeignKey(attribute='requester', replace_none_with_robot=True)
    responsibility = apifields.GroupResponsibilityForeignKey(full_list=False)
    role = apifields.RoleForeignKey(full_list=False)
    role_alias = apifields.RoleAliasForeignKey(full_list=False)
    role_field = apifields.RoleFieldForeignKey(full_list=False)
    role_node = apifields.RoleNodeForeignKey(full_list=False, attribute='role_node')
    rolerequest = apifields.RoleRequestForeignKey(full_list=False)
    workflow = apifields.WorkflowForeignKey(full_list=False)

    # Дополнительные поля
    ref_count = fields.IntegerField('ref_count', null=True, blank=True)

    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = Action
        resource_name = 'actions'
        allowed_methods = ['get', 'post']
        fields = [
            'id',
            'action', 'added', 'data', 'error',
            'role', 'user', 'group', 'system', 'role_node', 'role_field', 'role_alias', 'node_responsibility',
            'inconsistency', 'service', 'transfer', 'workflow',
            'rolerequest', 'approverequest',
            'requester', 'impersonator',
            'membership', 'responsibility',
            'passport_login',
            'parent',
        ]
        ordering = ['-id']
        limit = 100

    @staticmethod
    def get_current_system(request):
        form = forms.ActionForm(request.GET)
        if not form.is_valid():
            raise BadRequest(form.errors)

        system = form.cleaned_data.get('system')
        if system:
            return system
        role = form.cleaned_data.get('role')
        if role:
            return role.system
        return None

    def get_object_list(self, request, **kwargs):
        system = self.get_current_system(request)
        requester = self.get_requester(request)
        foreign_keys = [
            'group', 'system', 'user', 'approverequest', 'impersonator', 'inconsistency', 'membership',
            'requester', 'responsibility', 'role', 'role_alias', 'role_field', 'role_node', 'rolerequest',
            'workflow',
        ]
        return Action.objects.permitted_for(requester, system).order_by('-added', '-id').select_related(*foreign_keys)

    def get_object_list_for_detail(self, request, **kwargs):
        fields_for_join = [  # делаем select_related
            'requester', 'impersonator', 'role',
            'role__user', 'role__group', 'role__system', 'role__node', 'role__node__system',
            'workflow', 'workflow__system', 'workflow__user', 'workflow__approver',
            'membership', 'membership__user', 'membership__group',
            'responsibility', 'responsibility__user', 'responsibility__group',
            'rolerequest',
            'rolerequest__role', 'rolerequest__role__user', 'rolerequest__role__group', 'rolerequest__role__system',
            'rolerequest__role__node', 'rolerequest__requester',
            'approverequest', 'approverequest__approver',
            'role_node', 'role_node__system',
            'role_field',
            'role_field__node',
            'role_alias', 'role_alias__node'
        ]
        fields_for_prefetch = [  # делаем prefetch_related (в основном потому, что все поля не влезают в один JOIN)
            'inconsistency', 'inconsistency__system', 'inconsistency__user', 'inconsistency__group',
            'inconsistency__role__node', 'inconsistency__role__system', 'inconsistency__role__node__system',
            'inconsistency__role__user', 'inconsistency__node__system',
        ]
        return (
            self
            .get_object_list(request)
            .select_related(*fields_for_join)
            .prefetch_related(*fields_for_prefetch)
            .annotate(ref_count=Count('role__refs'))
        )

    def build_filters(self, request, filters=None):
        fix_user_field_filters(filters, ['user'])
        form = forms.ActionForm(filters)
        if not form.is_valid():
            raise BadRequest(form.errors)

        query = form.cleaned_data
        qset_filters = Q()

        if query['id']:
            qset_filters &= Q(id=query['id'])

        if filters.get('user'):
            user_ids = {user.pk for user in query['user']}
            if query['user_type']:
                user_q = Q(user_id__in=user_ids, user__type=query['user_type'])
            else:
                user_q = Q(user_id__in=user_ids)
            qset_filters &= (
                Q(impersonator_id__in=user_ids) |
                Q(requester_id__in=user_ids) |
                user_q
            )
        elif query['user_type']:
            qset_filters &= (Q(user__type=query['user_type']) | Q(user_id=None))

        if filters.get('group'):
            group_ids = {group.pk for group in query['group']}
            qset_filters &= Q(group_id__in=group_ids)

        if query['role']:
            qset_filters &= Q(role=query['role'])

        if query['path'] and query['path'].level > 0:
            qset_filters &= (
                Q(role__node__value_path__startswith=query['path'].value_path) |
                Q(role_node__value_path__startswith=query['path'].value_path)
            )

        if query['system']:
            qset_filters &= Q(system=query['system'])

        if query['action']:
            qset_filters &= Q(action__in=query['action'])

        if query['is_active'] is not None:
            qset_filters &= Q(role__is_active=query['is_active'])

        if query['date_from']:
            qset_filters &= Q(added__gte=query['date_from'])

        if query['date_to']:
            qset_filters &= Q(added__lt=query['date_to'])

        return qset_filters

    def apply_filters(self, request, applicable_filters, **kwargs):
        return self.get_object_list(request).filter(applicable_filters)

    def dehydrate(self, bundle):
        bundle.data['human_verb'] = bundle.obj.as_verb()
        bundle.data['human_noun'] = bundle.obj.as_noun()
        # TODO: remove after deprecation
        bundle.data['human'] = bundle.data['human_verb']
        bundle.data['error'] = bundle.data.get('error')
        bundle.data['comment'] = bundle.obj.comment

        return bundle

    def dehydrate_for_detail(self, bundle):
        if bundle.obj.action == 'resolve_inconsistency':
            try:
                if bundle.obj.inconsistency and bundle.obj.inconsistency.user:
                    if bundle.obj.inconsistency.type == bundle.obj.inconsistency.TYPE_OUR:
                        template = 'У %s отсутствует роль "%s" в системе %s, которая есть в IDM.'
                        role = bundle.obj.inconsistency.node.humanize()
                    else:
                        template = 'У %s обнаружена роль "%s" в системе %s, которая отсутствует в IDM.'
                        role = bundle.obj.inconsistency.node.humanize()

                    bundle.data['inconsistency_comment'] = template % (
                        inflect('кого', bundle.obj.inconsistency.user.get_full_name(), fio=True),
                        role,
                        bundle.obj.inconsistency.system.get_name(),
                    )
            except Exception:
                pass

        return bundle

    def post_list(self, request, **kwargs):
        data = self.deserialize(request, request.body)
        form = forms.ReportPostForm(data)
        if not form.is_valid():
            raise BadRequest(form.errors)

        filters = self.build_filters(request, data)
        actions = self.apply_filters(request, filters).select_related('role__node', 'role__system')
        requester = self.get_requester(request)

        make_report(actions, form.cleaned_data['format'], 'actions', requester.impersonated,
                    form.cleaned_data.get('comment', ''))

        return self.create_response(request, {
            'message': _('Отчёт формируется и будет отправлен вам на почту по завершении')
        })
