# coding: utf-8


from django.db.models import Q
from tastypie.bundle import Bundle

from idm.api.exceptions import BadRequest
from idm.api.frontend.base import FrontendApiResource
from idm.api.frontend.forms import AllApproversForm
from idm.core.models import ApproveRequest


class AllApproversResource(FrontendApiResource):
    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = ApproveRequest
        resource_name = 'all_approvers'
        list_allowed_methods = ['get']
        detail_allowed_methods = []

        select_related_for_approverequest_list = (
            'approver',
        )
        queryset = ApproveRequest.objects.select_related(*select_related_for_approverequest_list)

    def build_filters(self, request, filters=None):
        form = AllApproversForm(filters)

        if not form.is_valid():
            raise BadRequest(form.errors)

        approve = form.cleaned_data['approve_id']
        return Q(approve=approve)

    def apply_filters(self, request, applicable_filters, **kwargs):
        return self.get_object_list(request).filter(applicable_filters)

    def dehydrate(self, bundle: Bundle) -> Bundle:
        bundle.data = {
            'approved': bundle.obj.approved,
            'decision': bundle.obj.decision,
            'approver': {
                'username': bundle.obj.approver.username,
                'is_active': bundle.obj.approver.is_active,
                'full_name': bundle.obj.approver.get_full_name(),
            }
        }

        return bundle

