# coding: utf-8


from django.db.models import ObjectDoesNotExist
from django.utils.translation import ugettext as _
from tastypie.bundle import Bundle
from tastypie.fields import ApiField, ForeignKey

from idm.users.models import User


class ForRelated(object):
    """Boolean-like объект для определения дегидрации ресурсов
    """
    def __bool__(self):
        return True


AllButRelated = lambda bundle: not bundle.for_related  # noqa


class FilterableMixin(object):
    def __init__(self, *args, **kwargs):
        to_field = kwargs.pop('to_field', None)
        if to_field:
            self.to_field = to_field
        super(FilterableMixin, self).__init__(*args, **kwargs)


class DehydratedForListMixin(object):
    """Миксин, который вызывает full_dehydrate у related объекта с флагом for_list=True
    """
    def dehydrate_related(self, bundle, related_resource, for_list=True):
        """
        Based on the ``full_resource``, returns either the endpoint or the data
        from ``full_dehydrate`` for the related resource.
        """
        should_dehydrate_full_resource = self.should_full_dehydrate(bundle, for_list=for_list)

        if not should_dehydrate_full_resource:
            # Be a good netizen.
            return related_resource.get_resource_uri(bundle)
        else:
            # ZOMG extra data and big payloads.
            bundle = related_resource.build_bundle(
                obj=bundle.obj,
                request=bundle.request,
                objects_saved=bundle.objects_saved
            )
            return related_resource.full_dehydrate(bundle, for_list=ForRelated())


class BaseForeignKey(FilterableMixin, DehydratedForListMixin, ForeignKey):
    attribute = None
    resource = None
    api = None
    to_field = 'id'

    def __init__(self, *args, **kwargs):
        defaults = {
            'to': '{}.{}'.format(self.api, self.resource),
            'attribute': self.attribute,
            'null': True,
            'blank': True,
            'full': True,
            'to_field': self.to_field
        }
        defaults.update(kwargs)
        super(BaseForeignKey, self).__init__(*args, **defaults)


class BaseIntranetForeignKey(BaseForeignKey):
    api = 'idm.api.frontend'


class UserForeignKey(BaseIntranetForeignKey):
    resource = 'user.UserResource'
    attribute = 'user'
    to_field = 'username'

    def __init__(self, *args, **kwargs):
        self.replace_none_with_robot = kwargs.pop('replace_none_with_robot', False)
        super(UserForeignKey, self).__init__(*args, **kwargs)

    def dehydrate(self, bundle, for_list=True):
        result = super(UserForeignKey, self).dehydrate(bundle, for_list)

        if result is None and self.replace_none_with_robot:
            foreign_obj = User.objects.get_idm_robot()
            self.fk_resource = self.get_related_resource(foreign_obj)
            fk_bundle = Bundle(obj=foreign_obj, request=bundle.request)
            return self.dehydrate_related(fk_bundle, self.fk_resource, for_list=for_list)

        return result


class GroupForeignKey(BaseIntranetForeignKey):
    resource = 'group.GroupResource'
    attribute = 'group'
    to_field = 'slug'


class SystemForeignKey(BaseIntranetForeignKey):
    attribute = 'system'
    resource = 'system.SystemResource'


class ServiceForeignKey(BaseIntranetForeignKey):
    attribute = 'service'
    resource = 'service.ServiceResource'


class ServiceNoRecursionForeignKey(BaseIntranetForeignKey):
    attribute = 'service'
    resource = 'service.ServiceNoRecursionResource'


class ParentlessRoleForeignKey(BaseIntranetForeignKey):
    attribute = 'role'
    resource = 'role.ParentlessRoleResource'


class RoleForeignKey(BaseIntranetForeignKey):
    attribute = 'role'
    resource = 'role.FrontendRoleResource'


class RoleRequestForeignKey(BaseIntranetForeignKey):
    attribute = 'rolerequest'
    resource = 'rolerequest.RoleRequestResource'


class RoleRequestShortForeignKey(BaseIntranetForeignKey):
    attribute = 'rolerequest'
    resource = 'rolerequest.RoleRequestShortResource'


class RoleRequestForRoleForeignKey(BaseIntranetForeignKey):
    attribute = 'last_request'
    resource = 'rolerequest.RoleRequestForRoleResource'


class ApproveRequestForeignKey(BaseIntranetForeignKey):
    attribute = 'approverequest'
    resource = 'approverequest.ApproveRequestResource'


class InconsistencyForeignKey(BaseIntranetForeignKey):
    attribute = 'inconsistency'
    resource = 'inconsistency.InconsistencyResource'


class MembershipForeignKey(BaseIntranetForeignKey):
    attribute = 'membership'
    resource = 'membership.MembershipResource'


class WorkflowForeignKey(BaseIntranetForeignKey):
    attribute = 'workflow'
    resource = 'workflow.WorkflowResource'


class RoleNodeForeignKey(BaseIntranetForeignKey):
    attribute = 'node'
    resource = 'rolenode.RoleNodeResource'


class RoleAliasForeignKey(BaseIntranetForeignKey):
    attribute = 'role_alias'
    resource = 'rolealias.RoleAliasResource'


class RoleFieldForeignKey(BaseIntranetForeignKey):
    attribute = 'role_field'
    resource = 'rolefield.RoleFieldResource'


class GroupResponsibilityForeignKey(BaseIntranetForeignKey):
    attribute = 'groupresponsibility'
    resource = 'groupresponsibility.GroupResponsibilityResource'


class UserPassportLoginForeignKey(BaseIntranetForeignKey):
    attribute = 'passport_login'
    resource = 'passport_login.PassportLoginResource'

    def dehydrate(self, bundle, for_list=True):
        result = super(UserPassportLoginForeignKey, self).dehydrate(bundle, for_list)

        if not result:
            return result

        result_fields = ['id', 'login']
        result.data = {field: result.data[field] for field in result_fields}

        return result


class ApproverApiField(ApiField):
    """
    Представление подтверждающего запрос пользователя в API.
    Отличается от обычного пользователя тем, что не обрабатывает значение None
    и не предполагает, что аппрувер может быть None.
    """
    to_field = 'username'
    dehydrated_type = 'dict'
    help_text = 'Approver object'

    def convert(self, value):
        return {
            'username': value.username,
            'full_name': value.get_full_name(),
        }


class ApiUserField(ApiField):
    """
    Представление объекта Пользователя в API
    """
    to_field = 'username'
    dehydrated_type = 'dict'
    help_text = 'User object'

    def convert(self, value):
        if not value:
            return {
                'username': 'robot-idm',
                'full_name': _('Робот'),
            }

        return {
            'username': value.username,
            'full_name': value.get_full_name(),
        }


class ApiJSONField(ApiField):
    """
    Представление кастомного JSONField в API
    """
    dehydrated_type = 'dict'
    help_text = 'json-serialized data'

    def convert(self, value):
        if value is None or value == '':
            return None

        return value
