# coding: utf-8


import logging

from django.conf.urls import url
from django.db.models import Q
from django.shortcuts import get_object_or_404
from django.utils.translation import ugettext_lazy as _
from tastypie import fields
from tastypie.bundle import Bundle
from tastypie.utils import trailing_slash

from idm.api.exceptions import BadRequest
from idm.api.frontend import apifields
from idm.api.frontend.base import FrontendApiResource, FilterPermissionsMixin
from idm.api.frontend.forms import ApproveRequestForm, RoleApproveForm, ApproverForm, ApproveRequestBulkForm
from idm.api.frontend.requesthelpers import convert_role_request_exceptions
from idm.api.frontend.utils import fix_user_field_filters
from idm.core import exceptions
from idm.core.constants.approverequest import APPROVEREQUEST_DECISION, PriorityType
from idm.core.models import ApproveRequest, RoleNode
from idm.core.tasks.approverequest import ApproveRequestBulk

log = logging.getLogger(__name__)


class BaseApproveRequestResource(FilterPermissionsMixin, FrontendApiResource):
    approverequest_form_class = ApproveRequestForm

    approver = apifields.UserForeignKey(attribute='approver')
    is_done = fields.BooleanField(attribute='approve__role_request__is_done')
    requester = apifields.UserForeignKey(attribute='approve__role_request__requester', replace_none_with_robot=True)
    role = apifields.RoleForeignKey(attribute='approve__role_request__role')
    role_request = apifields.RoleRequestShortForeignKey(attribute='approve__role_request', full=True)

    filter_by_status_mapping = {
        approverequest_form_class.STATUS_ALL: Q(),
        approverequest_form_class.STATUS_PENDING: (
                Q(approve__approved__isnull=True, approve__role_request__is_done=False)
                & ~Q(decision=APPROVEREQUEST_DECISION.IGNORE)
        ),
        approverequest_form_class.STATUS_PROCESSED: (
                ~Q(approve__approved__isnull=True, approve__role_request__is_done=False)
                | Q(decision=APPROVEREQUEST_DECISION.IGNORE)
        ),
    }

    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = ApproveRequest
        resource_name = 'approverequests'
        list_allowed_methods = ['get']
        detail_allowed_methods = ['get', 'post']
        fields = ['id', 'approver', 'approved', 'decision', 'is_done', 'requester', 'role', 'role_request', 'requester']

    def get_object_list(self, request, requester_is_approver=False, **kwargs):
        """
        Переопределено, чтобы не создавать queryset в import-time.
        Возвращает queryset объектов запросов ролей, подтвердить которые должен аутентифицированный пользователь.
        """
        requester = self.get_requester(request)

        if requester_is_approver:
            approve_requests = ApproveRequest.objects.all()
        else:
            approve_requests = ApproveRequest.objects.permitted_for(requester)

            if request.method == 'GET':
                form = ApproverForm(request.GET)
                if not form.is_valid():
                    raise BadRequest(form.errors)

                approvers = form.cleaned_data['approver']
                if len(approvers) == 1 and approvers[0] == requester.impersonated:
                    approve_requests = ApproveRequest.objects.all()

        approve_requests = (
            approve_requests
            .filter(parent=None)  # выбираем только главные approve request-ы
            .select_related(
                'approver',
                'approve',
                'approve__role_request',
                'approve__role_request__requester',
                'approve__role_request__role',
                'approve__role_request__role__system',
                'approve__role_request__role__user',
                'approve__role_request__role__group',
                'approve__role_request__role__group__parent',
                'approve__role_request__role__node',
                'approve__role_request__role__node__nodeset',
                'approve__role_request__role__node__system',
                'approve__role_request__role__parent',
                'approve__role_request__role__parent__system',
                'approve__role_request__role__parent__user',
                'approve__role_request__role__parent__group',
                'approve__role_request__role__parent__node',
                'approve__role_request__role__parent__node__nodeset',
                'approve__role_request__role__parent__node__system',
            )
            .prefetch_related(
                'approve__role_request__approves__requests__approver',
            )
            .order_by('-added', '-pk'))

        return approve_requests

    def get_object_list_for_detail(self, request, **kwargs):
        requester_is_approver = False

        pk = kwargs['pk']
        assert pk is not None

        approverequest = get_object_or_404(ApproveRequest, pk=pk)
        requester = self.get_requester(request)
        if approverequest.approver_id == requester.impersonated.id:
            requester_is_approver = True

        return self.get_object_list(request, requester_is_approver=requester_is_approver, **kwargs)

    def filter_by_status(self, status, qset_filters):
        """вернуть все запросы без фильтров по статусу"""

        return qset_filters & self.filter_by_status_mapping[status]

    def filter_by_priority_type(self, priority_type, qset_filters):
        """вернуть все запросы без фильтров по типу приоритета"""

        # При пробросе priority_type_filter он становится списком, т.е. filters это QueryDict
        if isinstance(priority_type, list):
            priority_type = priority_type[0]

        return qset_filters & PriorityType(priority_type).queryset_filter

    def build_filters(self, request, filters=None):
        fix_user_field_filters(filters, ['user'])
        form = self.approverequest_form_class(filters)
        if not form.is_valid():
            raise BadRequest(form.errors)

        query = form.cleaned_data
        qset_filters = Q()
        permissions_params = {}

        if query['role_ids']:
            qset_filters &= Q(approve__role_request__role__in=query['role_ids'])

        priority_type = query['priority_type']
        if not priority_type:
            priority_type = filters.pop('priority_type_filter', None)

        if priority_type:
            qset_filters = self.filter_by_priority_type(priority_type, qset_filters)

        if query['status'] and filters.pop('status_filter', True) in (True, [True]):
            qset_filters = self.filter_by_status(query['status'], qset_filters)

        if query['reason']:
            qset_filters &= Q(approve__role_request__reason=query['reason'])

        if query['requester']:
            qset_filters &= Q(approve__role_request__requester__in=query['requester'])

        if query['approver']:
            if len(query['approver']) == 1:
                permissions_params['approver'] = query['approver']

            qset_filters &= Q(approver__in=[approver.id for approver in query['approver']])

        if filters.get('user') or filters.get('group'):
            # Роли пользователей и групп фильтруются через ИЛИ
            or_q = Q()

            if filters.get('user'):
                user_q = Q(approve__role_request__role__user__in=[user.id for user in query['user']])
                if query['user_type']:
                    user_q &= Q(approve__role_request__role__user__type=query['user_type'])
                or_q |= user_q

            if filters.get('group'):
                or_q |= Q(approve__role_request__role__group__in=[group.id for group in query['group']])

            qset_filters &= or_q
        elif query['user_type']:
            permissions_params['user_type'] = query['user_type']
            qset_filters &= Q(approve__role_request__role__user__type=query['user_type'])

        if query['system']:
            permissions_params['system'] = query['system']
            qset_filters &= Q(approve__role_request__role__system=query['system'])

            if query['nodeset']:
                qset_filters &= Q(approve__role_request__role__node__nodeset__in=query['nodeset'])

        if isinstance(query['path'], RoleNode):
            qset_filters &= Q(approve__role_request__role__node__rolenodeclosure_parents__parent=query['path'])

        return qset_filters, permissions_params

    def apply_filters(self, request, applicable_filters, **kwargs):
        return self.get_object_list(request).filter(applicable_filters)

    def dehydrate(self, bundle):
        """
        Привести объект запроса роли к подходящиему для фронтенда виду.
        """
        role_request = bundle.obj.approve.role_request

        bundle.data['approved'] = bundle.obj.approved
        # Статус запроса
        if bundle.obj.decision == APPROVEREQUEST_DECISION.NOT_DECIDED and not role_request.is_done:
            bundle.data['status'] = 'pending'
            bundle.data['human_status'] = _('Ожидает подтверждения')
        else:
            bundle.data['status'] = 'processed'
            bundle.data['human_status'] = _('Обработан')
        return bundle


class ApproveRequestResource(BaseApproveRequestResource):
    """
    Ресурс запросов на получение ролей пользователями
    """

    class Meta(BaseApproveRequestResource.Meta):
        """Всё как у родительского класса"""

    def dehydrate(self, bundle):
        bundle = super(ApproveRequestResource, self).dehydrate(bundle)
        role_request = bundle.obj.approve.role_request
        bundle.data['reason'] = role_request.get_reason()
        bundle.data['human_reason'] = role_request.get_reason_display()
        bundle.data['comment'] = role_request.comment
        return bundle

    def post_detail(self, request, pk, **kwargs):
        """
        Подтвердить, отказать, игнорировать или обсудить запрос
        """
        data = self.deserialize(request, request.body)
        form = RoleApproveForm(data)
        if not form.is_valid():
            raise BadRequest(form.errors)

        requester = self.get_requester(request, data)
        data = form.cleaned_data

        queryset = (
            ApproveRequest.objects
            .filter(approver=requester.impersonated)
            .select_related(
                'approve__role_request__role__system',
                'approve__role_request__role__node',
                'approve__role_request__requester',
                'approver'
            )
        )
        approve_request = get_object_or_404(queryset, pk=pk)

        decision = data['decision']
        assert decision in [
            APPROVEREQUEST_DECISION.APPROVE,
            APPROVEREQUEST_DECISION.DECLINE,
            APPROVEREQUEST_DECISION.IGNORE,
            APPROVEREQUEST_DECISION.DISCUSS,
        ]
        status_code = 204
        if decision == APPROVEREQUEST_DECISION.APPROVE:
            method = approve_request.set_approved
        elif decision == APPROVEREQUEST_DECISION.DECLINE:
            method = approve_request.set_declined
        elif decision == APPROVEREQUEST_DECISION.IGNORE:
            method = approve_request.set_ignored
        elif decision == APPROVEREQUEST_DECISION.DISCUSS:
            method = approve_request.start_discussion
            status_code = 201

        with convert_role_request_exceptions():
            try:
                result = method(
                    requester=requester,
                    comment=data['comment'],
                    from_api=True,
                )
            except exceptions.RoleStateSwitchError:
                # роль нельзя было подтверждать, т.к. она уже в другом состоянии.
                # просто не обращаем внимания
                result = None
        data = {'result': result} if status_code == 201 else None
        return self.create_response(request, data, status=status_code)

    def prepend_urls(self):
        url_template = r'^(?P<resource_name>{resource_name})/{url_part}{slash}$'

        def prefix_url(url_part):
            result = url_template.format(
                resource_name=self._meta.resource_name,
                slash=trailing_slash(),
                url_part=url_part
            )
            return result

        return [url(prefix_url('bulk'), self.wrap_view('approve_bulk'), name='api_approve_bulk')]

    def approve_bulk(self, request, **kwargs):
        data = self.deserialize(request, request.body)
        form = ApproveRequestBulkForm(data)
        raise BadRequest()
        if not form.is_valid():
            raise BadRequest(form.errors)

        assert form.cleaned_data['decision'] in [
            APPROVEREQUEST_DECISION.APPROVE,
            APPROVEREQUEST_DECISION.DECLINE,
            APPROVEREQUEST_DECISION.IGNORE,
        ]
        ApproveRequestBulk.delay(
            requester_pk=request.user.pk,
            priority_type=form.cleaned_data['priority_type'],
            decision=form.cleaned_data['decision'],
            username=data.get('user'),
            system_slug=data.get('system'),
            path=data.get('path'),
            role_ids=form.cleaned_data['role_ids'],
            comment=form.cleaned_data['comment'],
            chunk_size=1000,
        )
        return self.create_response(request, None, status=204)


class ApproveRequestNoRecursionResource(FrontendApiResource):
    def __init__(self, api_name: str = None):
        if api_name is not None:
            self._meta.api_name = api_name

        self.fields = {}

    def dehydrate(self, bundle: Bundle) -> Bundle:
        bundle.data['is_done'] = bundle.obj.approve.role_request.is_done
        bundle.data['approved'] = bundle.obj.approved
        bundle.data['decision'] = bundle.obj.decision

        approver = bundle.obj.approver
        bundle.data['approver'] = {
            'username': approver.username,
            'is_active': approver.is_active,
            'full_name': approver.get_full_name(),
        }

        return bundle
