# coding: utf-8
import types
from io import BytesIO

from django.conf import settings
from django.core.handlers.wsgi import WSGIRequest
from django.urls.resolvers import URLResolver, Resolver404
from django.utils.http import urlencode
from tastypie import http

from idm.api.frontend.batch.utils import override_settings
from idm.wsgi import IdmWSGIHandler
from idm.api.frontend.batch.response import ApiResponse


class RequestProcessor(IdmWSGIHandler):
    def __init__(self, original_request, resource, options=None):
        super(RequestProcessor, self).__init__()
        self._resource = resource
        self._original_request = original_request
        self.options = options

        with override_settings(MIDDLEWARE=settings.BATCH_MIDDLEWARE):
            self.load_middleware()

    def _get_resolver_match(self, path):
        path = "{}/{}/".format(self._resource.api_name, path.strip('/'))

        for resolver in self._resource.api.urls:
            if not isinstance(resolver, URLResolver):
                continue

            try:
                return resolver.resolve(path)
            except Resolver404:
                pass

    def _make_wsgi_request(self, api_request):
        env = self._original_request.META.copy()

        encoded_data = api_request.encoded_body
        env['REQUEST_METHOD'] = api_request.method.upper()
        env['PATH_INFO'] = api_request.path
        env['CONTENT_LENGTH'] = str(len(encoded_data))
        env['IS_BATCH_SUBREQUEST'] = True
        if env['REQUEST_METHOD'] == 'GET' and api_request.body:
            env['QUERY_STRING'] = urlencode(api_request.body, doseq=True)

        else:
            env['wsgi.input'] = BytesIO(encoded_data)

        subrequest = WSGIRequest(env)
        # чтобы не ходить в каждом подзапросе в blackbox
        subrequest._yauser = self._original_request._yauser

        return subrequest

    def _create_error_response(self, api_request, response_class, data):
        return ApiResponse(
            api_request=api_request,
            response=self._resource.create_response(
                request=self._original_request,
                response_class=response_class,
                data=data
            ),
        )

    def _get_response(self, request):
        """
        Resolve and call the view, then apply view, exception, and
        template_response middleware. This method is everything that happens
        inside the request/response middleware.
        """
        response = None

        resolver_match = self._get_resolver_match(request.path_info)
        callback, callback_args, callback_kwargs = resolver_match
        request.resolver_match = resolver_match

        # Apply view middleware
        for middleware_method in self._view_middleware:
            response = middleware_method(request, callback, callback_args, callback_kwargs)
            if response:
                break

        if response is None:
            wrapped_callback = self.make_view_atomic(callback)
            try:
                response = wrapped_callback(request, *callback_args, **callback_kwargs)
            except Exception as e:
                response = self.process_exception_by_middleware(e, request)

        # Complain if the view returned None (a common error).
        if response is None:
            if isinstance(callback, types.FunctionType):  # FBV
                view_name = callback.__name__
            else:  # CBV
                view_name = callback.__class__.__name__ + '.__call__'

            raise ValueError(
                "The view %s.%s didn't return an HttpResponse object. It "
                "returned None instead." % (callback.__module__, view_name)
            )

        # If the response supports deferred rendering, apply template
        # response middleware and then render the response
        elif hasattr(response, 'render') and callable(response.render):
            for middleware_method in self._template_response_middleware:
                response = middleware_method(request, response)
                # Complain if the template response middleware returned None (a common error).
                if response is None:
                    raise ValueError(
                        "%s.process_template_response didn't return an "
                        "HttpResponse object. It returned None instead."
                        % (middleware_method.__self__.__class__.__name__)
                    )

            try:
                response = response.render()
            except Exception as e:
                response = self.process_exception_by_middleware(e, request)

        return response

    def process_api_request(self, api_request):
        if not api_request.is_valid():
            return self._create_error_response(
                api_request=api_request,
                response_class=http.HttpBadRequest,
                data={
                    'message': api_request.error_message,
                    'error_code': 'BAD_REQUEST',
                },
            )

        match = self._get_resolver_match(api_request.path)
        if not match:
            return self._create_error_response(
                api_request=api_request,
                response_class=http.HttpNotFound,
                data={
                    'message': 'path not found',
                    'error_code': 'NOT_FOUND',
                },
            )

        if match.kwargs.get('resource_name') == self._resource._meta.resource_name:
            return self._create_error_response(
                api_request=api_request,
                response_class=http.HttpForbidden,
                data={
                    'message': 'batch recursion is not allowed',
                    'error_code': 'FORBIDDEN',
                },
            )

        wsgi_request = self._make_wsgi_request(api_request)
        response = self.get_response(wsgi_request)

        return ApiResponse(api_request, response)
