# coding: utf-8


import logging

from idm.api.exceptions import BadRequest
from idm.api.frontend import apifields, forms
from idm.api.frontend.base import FrontendApiResource
from idm.api.frontend.paginator import DummyPaginator
from idm.core.models import FavoriteSystem

log = logging.getLogger(__name__)


class FavoriteSystemResource(FrontendApiResource):
    """
    Ресурс системы в избранном
    """
    system = apifields.SystemForeignKey()

    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = FavoriteSystem
        resource_name = 'favoritesystems'
        list_allowed_methods = ['get', 'post', 'delete']
        detail_allowed_methods = []
        paginator_class = DummyPaginator
        fields = ['id', 'added', 'system']

    def get_object_list(self, request, **kwargs):
        return FavoriteSystem.objects.filter(user=request.user).select_related('system')

    def post_list(self, request, **kwargs):
        data = self.deserialize(request, request.body)
        form = forms.FavoriteSystemForm(data)
        if not form.is_valid():
            raise BadRequest(form.errors)

        fs, created = FavoriteSystem.objects.get_or_create(
            user=request.user,
            system=form.cleaned_data['system']
        )

        status = 201 if created else 204
        return self.create_response(request, None, status=status)

    def delete_list(self, request, **kwargs):
        data = self.deserialize(request, request.body)
        form = forms.FavoriteSystemForm(data)
        if not form.is_valid():
            raise BadRequest(form.errors)

        FavoriteSystem.objects.filter(user=request.user, system=form.cleaned_data['system']).delete()
        return self.create_response(request, None, status=204)
