import logging
from itertools import chain

import re
from typing import Optional

import pytz
from django import forms
from django.core.exceptions import ValidationError
from django.db.models import NOT_PROVIDED
from django.forms.models import ModelChoiceField, ModelMultipleChoiceField
from django.utils import six
from django.utils.encoding import smart_text, force_text
from django.utils.translation import ugettext_lazy as _
from django.utils import timezone
from ylog.context import log_context

from idm.core.models import System, Role, RoleNode, RoleNodeSet
from idm.core.constants.action import ACTION
from idm.core.constants.role import ROLE_STATE
from idm.services.models import Service
from idm.users.models import User, Group
from idm.utils import json
from idm.utils.i18n import get_lang_pair

log = logging.getLogger(__name__)


class LowerCharField(forms.CharField):
    def to_python(self, value):
        value = super(LowerCharField, self).to_python(value)
        if value:
            value = value.lower()

        return value


class MaybeWidget(forms.Widget):
    def value_from_datadict(self, data, files, name):
        return data.get(name, NOT_PROVIDED)


class LookupWidget(forms.Widget):
    def value_from_datadict(self, data, files, name):
        values = {}
        for param_name in data:
            match = re.match(('%s__([\w-]+)' % name), param_name)
            if match:
                values[match.group(1)] = data[param_name]
        return values


class LookupCharField(forms.Field):
    widget = LookupWidget


class MaybeJSONField(forms.Field):
    widget = MaybeWidget
    empty_values = forms.Field.empty_values + [NOT_PROVIDED]

    def clean(self, value):
        value = super(MaybeJSONField, self).clean(value)
        if value not in self.empty_values:
            try:
                value = json.loads(value)
            except ValueError:
                raise forms.ValidationError(_('Данные полей должны быть представлены в формате JSON'))
        return value


class InvalidJSONInput(six.text_type):
    pass


class JSONString(six.text_type):
    pass


class JSONField(forms.CharField):
    """
        Копипаста из django 1.11
    """

    default_error_messages = {
        'invalid': _("'%(value)s' value must be valid JSON."),
    }
    widget = forms.Textarea

    def to_python(self, value):
        if self.disabled:
            return value
        if value in self.empty_values:
            return None
        elif isinstance(value, (list, dict, int, float, JSONString)):
            return value
        try:
            converted = json.loads(value)
        except ValueError:
            raise forms.ValidationError(
                self.error_messages['invalid'],
                code='invalid',
                params={'value': value},
            )
        if isinstance(converted, six.text_type):
            return JSONString(converted)
        else:
            return converted

    def bound_data(self, data, initial):
        if self.disabled:
            return initial
        try:
            return json.loads(data)
        except ValueError:
            return InvalidJSONInput(data)

    def prepare_value(self, value):
        if isinstance(value, InvalidJSONInput):
            return value
        return json.dumps(value)


class NoValidationField(forms.Field):
    def clean(self, value):
        return value


class NullableCharField(forms.CharField):
    def to_python(self, value):
        # --- не хотим трогать None ---
        if value is None:
            return value
        # -----------------------------
        if value in self.empty_values:
            return ''
        return smart_text(value)


class BetterErrorMixin(object):
    """Миксин позволяющей пробрасывать в сообщение об ошибке
    проверяемое невалидное значение"""

    def to_python(self, value):
        try:
            result = super(BetterErrorMixin, self).to_python(value)
        except forms.ValidationError as exc:
            exc.params = {'value': value}
            raise exc
        return result


class CommaSeparatedMixin(object):
    """Миксин поддерживает ситуацию, когда нам переданы в одном
    параметре несколько значений через запятую
    """
    def _check_values(self, value):
        return super(CommaSeparatedMixin, self)._check_values(list(chain(*[str(v).split(',') for v in value])))


class CommaSeparatedMultipleChoiceField(forms.MultipleChoiceField):
    def to_python(self, value):
        if isinstance(value, six.string_types):
            value = value.split(',')

        value = super(CommaSeparatedMultipleChoiceField, self).to_python(value)

        return list(chain(*[v.split(',') for v in value]))


class CaseInsensitiveModelChoiceField(forms.ModelChoiceField):
    def to_python(self, value):
        if value in self.empty_values:
            return None
        value = value.lower()
        return super(CaseInsensitiveModelChoiceField, self).to_python(value)


class CommaSeparatedCharField(forms.CharField):
    def to_python(self, value):
        if value in self.empty_values:
            return ()
        if isinstance(value, six.string_types):
            value = value.split(',')
        result = [super(CommaSeparatedCharField, self).to_python(subvalue) for subvalue in value]
        return result


class CommaSeparatedIntegerField(forms.IntegerField):
    def to_python(self, value):
        if value in self.empty_values:
            return ()
        if isinstance(value, six.string_types):
            value = value.split(',')
        result = [super(CommaSeparatedIntegerField, self).to_python(subvalue) for subvalue in value]
        return result


class RoleStateField(CommaSeparatedMultipleChoiceField):
    def __init__(self, *args, **kwargs):
        kwargs['label'] = _('Состояния ролей')
        kwargs['choices'] = list(ROLE_STATE.STATE_CHOICES.items())
        super(RoleStateField, self).__init__(*args, **kwargs)


class UserField(BetterErrorMixin, CaseInsensitiveModelChoiceField):
    def __init__(self, **kwargs):
        defaults = {
            'queryset': User.objects.all(),
            'to_field_name': 'username',
            'error_messages': {
                'invalid_choice': _('Пользователь %(value)s не найден')
            }
        }
        defaults.update(kwargs)
        super(UserField, self).__init__(**defaults)


class UsersField(ModelMultipleChoiceField):
    def __init__(self, enforce_existence=True, **kwargs):
        defaults = {
            'queryset': User.objects.all(),
            'to_field_name': 'username',
        }
        defaults.update(kwargs)
        super(UsersField, self).__init__(**defaults)
        self.enforce_existence = enforce_existence

    def _check_values(self, value):
        # Копипаста CommaSeparatedMixin
        value = list(chain(*[v.split(',') for v in value]))
        # Копипаста оригинального метода из ModelMultipleChoiceField,
        # Но с возможностью отключить проверку существования объектов (для фильтров)
        key = self.to_field_name or 'pk'
        try:
            value = frozenset(value)
        except TypeError:
            raise ValidationError(
                self.error_messages['list'],
                code='list',
            )
        for pk in value:
            try:
                self.queryset.filter(**{key: pk})
            except (ValueError, TypeError):
                raise ValidationError(
                    self.error_messages['invalid_pk_value'],
                    code='invalid_pk_value',
                    params={'pk': pk},
                )
        qs = self.queryset.filter(**{'%s__in' % key: value})
        if self.enforce_existence:
            pks = set(force_text(getattr(o, key)) for o in qs)
            for val in value:
                if force_text(val) not in pks:
                    raise ValidationError(
                        self.error_messages['invalid_choice'],
                        code='invalid_choice',
                        params={'value': val},
                    )
        return qs


class GroupField(BetterErrorMixin, ModelChoiceField):
    def __init__(self, only_active=False, **kwargs):
        defaults = {
            'queryset': (Group.objects.active() if only_active else Group.objects).user_groups(),
            'to_field_name': 'external_id',
            'error_messages': {
                'invalid_choice': _('Группа с id=%(value)s не найдена')
            }
        }
        defaults.update(kwargs)
        super(GroupField, self).__init__(**defaults)


class GroupsField(CommaSeparatedMixin, ModelMultipleChoiceField):
    def __init__(self, only_active=False, **kwargs):
        defaults = {
            'queryset': (Group.objects.active() if only_active else Group.objects).user_groups(),
            'to_field_name': 'external_id',
        }
        defaults.update(kwargs)
        super(GroupsField, self).__init__(**defaults)


class PassportLoginField(forms.CharField):
    def clean(self, value):
        if value == '':
            return None
        return super(PassportLoginField, self).clean(value)


class ServicesField(CommaSeparatedMixin, ModelMultipleChoiceField):
    def __init__(self, **kwargs):
        kwargs.update({
            'queryset': Service.objects.all(),
            'to_field_name': 'external_id',
        })
        super(ServicesField, self).__init__(**kwargs)


class ServiceField(CommaSeparatedMixin, ModelChoiceField):
    def __init__(self, **kwargs):
        kwargs.update({
            'queryset': Service.objects.filter(parent__isnull=False),
            'to_field_name': 'external_id',
        })
        super(ServiceField, self).__init__(**kwargs)


class ServiceJSONField(ModelChoiceField):
    def __init__(self, **kwargs):
        kwargs.update({
            'queryset': Service.objects.all(),
            'to_field_name': 'external_id',
        })
        super(ServiceJSONField, self).__init__(**kwargs)

    def to_python(self, value):
        if value in self.empty_values:
            return None

        if not isinstance(value, dict):
            raise ValidationError('Service must be JSON')

        value = value.get('id')
        if not value:
            raise ValidationError('Service must contain id')

        return super(ServiceJSONField, self).to_python(value)


class TvmIdField(ModelChoiceField):
    def __init__(self, **kwargs):
        kwargs.update({
            'queryset': User.objects.active().tvm_apps(),
            'to_field_name': 'username',
        })
        super(TvmIdField, self).__init__(**kwargs)

    def to_python(self, value):
        if value in self.empty_values:
            return None

        return value


class SystemField(BetterErrorMixin, CaseInsensitiveModelChoiceField):
    def __init__(self, select_related=None, **kwargs):
        select_related = select_related or []
        kwargs.update({
            'queryset': System.objects.select_related(*select_related),
            'to_field_name': 'slug',
            'error_messages': {
                'invalid_choice': _('Система c slug=%(value)s не найдена')
            }
        })
        super(SystemField, self).__init__(**kwargs)


class RoleField(BetterErrorMixin, ModelChoiceField):
    def __init__(self, select_related=None, **kwargs):
        select_related = select_related or []
        kwargs.update({
            'queryset': Role.objects.select_related(*select_related),
            'to_field_name': 'id',
            'error_messages': {
                'invalid_choice': _('Роль c id=%(value)s не найдена')
            }
        })
        super(RoleField, self).__init__(**kwargs)


class SystemRequiredMixin(object):
    def __init__(self, **kwargs):
        self.system_field_name = kwargs.pop('system_field_name', 'system')

        super(SystemRequiredMixin, self).__init__(**kwargs)

    def get_system(self):
        if hasattr(self, 'system'):
            system = self.system
        else:
            system = self.form.cleaned_data.get(self.system_field_name)

        return system


class RoleNodeBaseField(SystemRequiredMixin, ModelChoiceField):
    def __init__(self, **kwargs):
        defaults = {
            'queryset': RoleNode.objects.active(),
            'error_messages': {
                'invalid_choice': _('Узел в дереве ролей не найден')
            }
        }
        kwargs.update(defaults)
        super(RoleNodeBaseField, self).__init__(**kwargs)

    def to_python(self, value) -> Optional[RoleNode]:
        if value in self.empty_values:
            return None

        system = self.get_system()

        if system:
            try:
                value = self.get_node(system, value)
            except (ValueError, self.queryset.model.DoesNotExist):
                with log_context(system=system.slug, field=self.__class__.__name__, log_name='node_does_not_exist'):
                    log.warning('%s: node %s does not exist', self.__class__.__name__, value)
                raise forms.ValidationError(self.error_messages['invalid_choice'], code='invalid_choice')
        else:
            value = None

        return value

    def get_node(self, system, value):
        raise NotImplemented


class RoleNodeValueField(RoleNodeBaseField):
    # /project/manager/
    def get_node(self, system, value):
        return self.queryset.model.objects.get_node_by_value_path(system, value)


class RoleNodeSlugField(RoleNodeBaseField):
    # /projects/project/role/manager/
    def get_node(self, system, value):
        return self.queryset.model.objects.get_node_by_slug_path(system, value)


class RoleNodeSetField(SystemRequiredMixin, CommaSeparatedMultipleChoiceField):
    queryset = RoleNodeSet.objects.active()

    def __init__(self, **kwargs):
        defaults = {
            'error_messages': {
                'invalid_choice': _('Набор ролей не найден')
            }
        }
        kwargs.update(defaults)
        super(RoleNodeSetField, self).__init__(**kwargs)

    def to_python(self, value):
        if value in self.empty_values:
            return []

        system = self.get_system()
        nodeset_ids = super(RoleNodeSetField, self).to_python(value)
        nodesets = []
        if system:
            for nodeset_id in nodeset_ids:
                try:
                    nodesets.append(self.queryset.filter(system=system, set_id=nodeset_id).get())
                except self.queryset.model.DoesNotExist:
                    raise forms.ValidationError(self.error_messages['invalid_choice'], code='invalid_choice')
        return nodesets

    def validate(self, value):
        return value


class ActionTypesField(BetterErrorMixin, CommaSeparatedMultipleChoiceField):
    def __init__(self, *args, **kwargs):
        kwargs['choices'] = ((k, v[2]) for k, v in ACTION.ACTIONS.items())
        super(ActionTypesField, self).__init__(*args, **kwargs)


class NullBooleanField(forms.CharField):
    def to_python(self, value):
        if value is None:
            return None

        if isinstance(value, bool):
            return value

        if value.lower() in ('true', '1'):
            return True
        if value.lower() in ('false', '0'):
            return False
        return None


class NullBooleanTrueField(NullBooleanField):
    def clean(self, value):
        value = super(NullBooleanTrueField, self).clean(value)
        return True if value is None else value


class LangField(forms.Field):
    def to_python(self, value):
        if value in self.empty_values:
            return None

        try:
            ru, en = get_lang_pair(value)
        except Exception:
            raise forms.ValidationError(_('Неверный формат поля'))

        return {'ru': ru, 'en': en}


class WarningField(LangField):
    def to_python(self, value):
        ret = super(WarningField, self).to_python(value)

        if isinstance(ret, dict):
            for key in ret:
                if not isinstance(ret[key], str):
                    raise forms.ValidationError(_('Неверный формат поля'))

        return ret


class NestedObjectField(forms.Field):
    def __init__(self, nested_form_class, *args, **kwargs):
        self.nested_form_class = nested_form_class
        super(NestedObjectField, self).__init__(*args, **kwargs)

    def to_python(self, value):
        if value is None:
            return []
        if not isinstance(value, list):
            raise forms.ValidationError(_('Ожидается список словарей'))

        cleaned = []
        for item in value:
            if not isinstance(item, dict):
                raise forms.ValidationError(_('Ожидается список словарей'))
            nested_form = self.nested_form_class(data=item)
            if not nested_form.is_valid():
                raise forms.ValidationError(nested_form.get_error_message())
            cleaned.append(nested_form.cleaned_data)
        return cleaned


class IterableField(forms.Field):
    def __init__(self, subfield_class, *args, **kwargs):
        self.subfield = subfield_class()
        super(IterableField, self).__init__(*args, **kwargs)

    def to_python(self, value):
        if value is None:
            return []

        if not hasattr(value, '__iter__'):
            raise forms.ValidationError(_('Поле должно поддерживать итерацию'))

        ret = []
        errors = []
        for item in value:
            try:
                ret.append(self.subfield.clean(item))
            except forms.ValidationError as e:
                errors.append((
                    item,
                    ', '.join(e.messages)
                ))

        if errors:
            raise forms.ValidationError(
                ', '.join(
                    '"{}" - {}'.format(item, error)
                    for item, error in errors
                )
            )

        return ret


class TZAwareDateTimeField(forms.DateTimeField):
    tz_format = re.compile('(?P<tz>\+[\d]{2}:[\d]{2})$')

    def __init__(self, input_formats=None, *args, **kwargs):
        super(TZAwareDateTimeField, self).__init__(input_formats=input_formats, *args, **kwargs)
        self.input_formats = list(self.input_formats)
        self.input_formats.extend([
            '%Y-%m-%d %H:%M:%S%z',
            '%Y-%m-%d %H:%M:%S.%f%z',
            '%Y-%m-%dT%H:%M:%S%z',
            '%Y-%m-%dT%H:%M:%S.%f%z',
        ])

    def strptime(self, value, format):
        tz = None
        if '%z' in format:
            match = self.tz_format.search(value)
            if match:
                tz_info = match.groups('tz')[0]
                tz = pytz.FixedOffset(
                    int('%s%s' % (tz_info[1:3], tz_info[4:]))
                )
                format = format.replace('%z', tz_info)
            else:
                raise TypeError('Value does not end with +XXXX')
        result = super(TZAwareDateTimeField, self).strptime(value, format)
        if tz is not None:
            result = timezone.make_aware(result, tz)
        return result


class StringListField(forms.CharField):
    def to_python(self, value):
        if not value:
            return None
        if isinstance(value, list):
            return ','.join([item.strip() for item in value])
        else:
            raise forms.ValidationError(_('List of strings expected'))
