import logging
import weakref
from datetime import datetime, time, timedelta
from typing import List

from django import forms
from django.conf import settings
from django.contrib.postgres.forms import SimpleArrayField
from django.core import validators
from django.utils import timezone
from django.utils.dateformat import DateFormat
from django.utils.translation import ugettext_lazy as _
from django.utils.functional import cached_property

from idm.api.frontend import fields
from idm.core.canonical import CanonicalField, CanonicalAlias, CanonicalResponsibility, CanonicalNode
from idm.core.constants.action import ACTION
from idm.core.constants.approverequest import APPROVEREQUEST_DECISION, PriorityType
from idm.core.constants.groupmembership import GROUPMEMBERSHIP_STATE
from idm.core.constants.role import ROLE_STATE, MAX_ROLE_TTL_DAYS
from idm.core.constants.rolefield import FIELD_TYPE
from idm.core.constants.system import (
    SYSTEM_AUTH_FACTOR,
    SYSTEM_STATE,
    SYSTEM_PLUGIN_TYPE,
    SYSTEM_NODE_PLUGIN_TYPE,
    SYSTEM_AUDIT_METHOD,
    SYSTEM_GROUP_POLICY,
    SYSTEM_REQUEST_POLICY,
    SYSTEM_ROLE_GRANT_POLICY,
    SYSTEM_INCONSISTENCY_POLICY,
    SYSTEM_PASSPORT_POLICY,
    SYSTEM_WORKFLOW_APPROVE_POLICY,
    SYSTEM_ROLETREE_POLICY,
    SYSTEM_REVIEW_ON_RELOCATE_POLICY,
)
from idm.core.constants.workflow import REQUEST_TYPE
from idm.core.models import RoleNode, RoleField, RoleAlias, Transfer, Approve, ApproveRequest, System, RoleRequest
from idm.core.plugins.generic import HANDLE_CHOICES, LIBRARY_CHOICES
from idm.framework.utils import add_to_instance_cache
from idm.inconsistencies.models import Inconsistency
from idm.users.constants.group import GROUP_TYPES
from idm.users.constants.user import USER_TYPES
from idm.users.models import Group, User
from idm.utils.i18n import get_lang_pair

log = logging.getLogger(__name__)


def prepare_date_field(value, error_message):
    if not value:
        return

    value = timezone.make_aware(
        datetime.combine(value, time(0, 0, 0)),
        timezone.get_current_timezone()
    )

    today = timezone.localtime(timezone.now())
    if value.date() < today.date():
        raise forms.ValidationError('%s: "%s".' % (error_message, DateFormat(value).format(settings.DATE_FORMAT)))
    # мы считаем, что все даты включительные
    value += timedelta(days=1)
    return value


class BaseForm(forms.Form):
    """Форма, умеющая по-разному работать со списком ошибок"""

    def get_error_message(self):
        messages = []
        for fieldname, error_list in self.errors.items():
            if fieldname == '__all__':
                messages.append(', '.join(error_list))
            else:
                label = self.fields[fieldname].label
                messages.append('%s: %s' % (label, ', '.join(error_list)))
        message = '; '.join(messages)
        return message


class SuggestForm(BaseForm):
    q = forms.CharField(label=_('Поисковый запрос'), required=False)
    id = forms.CharField(label=_('Id записи'), required=False)
    id__in = forms.CharField(label=_('Список Id записей'), required=False)
    offset = forms.IntegerField(
        label='Смещение',
        min_value=0,
        error_messages={
            'invalid': _('Ожидается число'),
            'min_value': _('Значение должно быть неотрицательным'),
        },
        required=False,
    )
    limit = forms.IntegerField(
        label='Лимит',
        min_value=0,
        error_messages={
            'invalid': _('Ожидается число'),
            'min_value': _('Значение должно быть неотрицательным'),
        },
        required=False,
    )
    type = forms.CharField(label=_('Тип'), required=False)

    def clean_q(self):
        return (self.cleaned_data['q'] or '').strip().lower()

    def clean_offset(self):
        return self.cleaned_data['offset'] or settings.IDM_SUGGEST_OFFSET

    def clean_limit(self):
        return self.cleaned_data['limit'] or settings.IDM_SUGGEST_LIMIT


class SuggestByRequiredUserForm(SuggestForm):
    user = fields.UserField(label=_('Пользователь'))


class SubjectFormMixin(forms.Form):
    user = fields.UserField(label=_('Пользователи'), required=False)
    group = fields.GroupField(label=_('Группа'), required=False, only_active=True)

    def clean(self):
        cleaned_data = super().clean()

        if self.errors:
            return

        if not (cleaned_data.get('user') or cleaned_data.get('group')):
            raise forms.ValidationError(_('Вы должны указать либо сотрудника, либо группу'))

        return cleaned_data

    def get_subject(self):
        subject = self.cleaned_data.get('group')
        if subject is None:
            subject = self.cleaned_data.get('user')
        return subject


class SuggestByRequiredSystemForm(SuggestForm):
    system = fields.SystemField(label=_('Система'))


class RolesSuggestForm(SuggestForm):
    system = fields.SystemField(label=_('Система'), select_related=['root_role_node'])
    scope = fields.RoleNodeValueField(label=_('Роль'), required=False)

    def __init__(self, *args, **kwargs):
        super(RolesSuggestForm, self).__init__(*args, **kwargs)
        self.fields['scope'].form = weakref.proxy(self)

    def clean(self):
        cleaned_data = super(RolesSuggestForm, self).clean()

        if cleaned_data.get('system') and cleaned_data.get('scope') is None:
            if 'scope' in self._errors:
                del self._errors['scope']
                raise forms.ValidationError('scope "%s" не найден в дереве ролей' % self.data['scope'])
            cleaned_data['scope'] = cleaned_data['system'].root_role_node
            add_to_instance_cache(cleaned_data['scope'], 'system', cleaned_data['system'])

        return cleaned_data


class UserRolesSuggestForm(RolesSuggestForm):
    user = fields.UserField(label=_('Пользователь'), required=False)


class PassportLoginRequestForm(BaseForm):
    system = fields.SystemField(label=_('Система'), required=False, select_related=['root_role_node'])
    path = fields.RoleNodeValueField(label=_('Роль'), required=False)

    def __init__(self, user, *args, **kwargs):
        self.user = user
        super(PassportLoginRequestForm, self).__init__(*args, **kwargs)
        self.fields['path'].form = weakref.proxy(self)

    def clean(self):
        data = self.data
        if 'path' in data and 'system' not in data:
            raise forms.ValidationError('"path" должен быть передан вместе с "system"')
        return super(PassportLoginRequestForm, self).clean()


class SystemForm(BaseForm):
    is_broken = forms.NullBooleanField()
    service = fields.ServicesField(label=_('Сервис'), required=False)
    responsibles = fields.UsersField(label=_('Ответственные'), required=False)
    team_members = fields.UsersField(label=_('Участники команды'), required=False)
    is_sox = fields.NullBooleanField(label=_('SOX'), required=False)
    is_favorite = forms.BooleanField(label=_('Избранность'), required=False)
    state = forms.ChoiceField(label=_('Статус'), choices=SYSTEM_STATE.CHOICES, required=False)
    use_webauth = forms.NullBooleanField(label=_('WebAuth'), required=False)
    system__contains = forms.CharField(label=_('Содержит подстроку'), required=False)


class CreateSystemForm(BaseForm):
    slug = forms.SlugField(label=_('Slug'), required=True)
    name = fields.LangField(label=_('Название'), required=True)
    description = fields.LangField(label=_('Описание'), required=False)
    service = fields.ServiceField(label=_('Сервис в ABC'), required=True)
    responsibles_user = fields.UsersField(label=_('Ответственные (пользователи)'), required=False)
    team_members_user = fields.UsersField(label=_('Участники команды (пользователи)'), required=False)
    responsibles_group = fields.GroupsField(label=_('Ответственные (группы)'), required=False)
    team_members_group = fields.GroupsField(label=_('Участники команды (группы)'), required=False)

    def clean_name(self):
        name = self.cleaned_data['name']
        for lang, local_name in name.items():
            name[lang] = local_name.strip()
        if not all(name.values()):
            self.add_error('name', _('Имя не может быть пустым'))
        return name

    def clean(self):
        cleaned_data = super(CreateSystemForm, self).clean()

        if self.errors:
            return

        cleaned_data['slug'] = cleaned_data['slug'].lower()

        if not cleaned_data['description']:
            cleaned_data['description'] = {'ru': '', 'en': ''}

        if System.objects.filter(slug=cleaned_data['slug']).exists():
            raise forms.ValidationError(_('System with slug "{}" already exists').format(cleaned_data['slug']))
        if not (cleaned_data['responsibles_user'] or cleaned_data['responsibles_group']):
            self.add_error('responsibles_user', _('Нет ответственных за систему'))

        return cleaned_data


class SoxSystemEditForm(BaseForm):
    sync_interval = forms.DurationField(required=False, label=_('Sync interval'))
    is_active = fields.NullBooleanField(required=False, label=_('Is active'))
    name = fields.LangField(required=False, label=_('Название'))
    description = fields.LangField(required=False, label=_('Описание'))
    group_policy = forms.ChoiceField(label=_('group policy'), required=False, choices=SYSTEM_GROUP_POLICY.CHOICES)
    service = fields.ServiceJSONField(required=False, label=_('service'))
    passport_policy = forms.ChoiceField(label=_('passport policy'), required=False,
                                        choices=SYSTEM_PASSPORT_POLICY.CHOICES)
    request_policy = forms.ChoiceField(label=_('request policy'), required=False, choices=SYSTEM_REQUEST_POLICY.CHOICES)
    role_grant_policy = forms.ChoiceField(label=_('role grant policy'), required=False,
                                          choices=SYSTEM_ROLE_GRANT_POLICY.CHOICES)
    roletree_policy = forms.ChoiceField(label=_('inconsistency policy'), required=False,
                                        choices=SYSTEM_ROLETREE_POLICY.CHOICES)
    tvm_id = fields.TvmIdField(required=False, label=_('tvm id'))
    check_certificate = fields.NullBooleanField(required=False, label=_('check certificate'))
    base_url = forms.CharField(required=False, label=_('base url'),
                               validators=[validators.URLValidator(schemes=('http', 'https'))])
    audit_method = forms.ChoiceField(required=False, label=_('audit method'), choices=SYSTEM_AUDIT_METHOD.CHOICES)
    plugin_type = forms.ChoiceField(
        label=_('plugin type'),
        required=False,
        choices=SYSTEM_PLUGIN_TYPE.CHOICES,
    )
    node_plugin_type = forms.ChoiceField(
        label=_('node plugin type'),
        required=False,
        choices=SYSTEM_NODE_PLUGIN_TYPE.CHOICES,
    )
    roles_tree_url = forms.CharField(required=False, label=_('roles tree url'),
                                     validators=[validators.URLValidator(schemes=('http', 'https'))])
    use_tvm_role = fields.NullBooleanField(label=_('use tvm role'), required=False)
    use_workflow_for_deprive = fields.NullBooleanField(label=_('use workflow for deprive'), required=False)
    emails = fields.StringListField(required=False, label=_('emails'))
    retry_failed_roles = forms.BooleanField(label=_('retry roles adding forever'), required=False)
    rolefields = fields.CommaSeparatedCharField(required=False, label=_('role fields data'))
    export_to_tirole = fields.NullBooleanField(required=False, label=_('Выгружать роли в Tirole'))
    tvm_tirole_ids = SimpleArrayField(fields.TvmIdField(), required=False, label=_('TVM ids for tirole'))

    def __init__(self, *args, requester=None, system=None, **kwargs):
        super(BaseForm, self).__init__(*args, **kwargs)
        if 'initial' not in kwargs and system:
            system.fetch_service()
            self.initial = {name: getattr(system, name) for name in self.fields if hasattr(system, name)}
        self.requester = requester
        self.system = system

    def check_if_changed(self, field, predicate, error_msg):
        value = self.cleaned_data[field]
        if field in self.changed_data and getattr(self.system, field) != value and not predicate(value):
            raise forms.ValidationError(error_msg)
        return value

    @cached_property
    def changed_data(self):
        return [name for name in super().changed_data if name in self.data]

    def clean_tvm_id(self):
        def valid(tvm_id: str):
            return self.requester.get_managed_tvm_apps().filter(username=tvm_id).exists()

        return self.check_if_changed(
            'tvm_id',
            valid,
            _('Можно указать только TVM-приложения, для которых вы являетесь управляющим в ABC'),
        )

    def clean_tvm_tirole_ids(self):
        def valid(tvm_tirole_ids: List[int]):
            return set(map(str, tvm_tirole_ids)) == set(
                self.requester
                    .get_managed_tvm_apps()
                    .filter(username__in=tvm_tirole_ids)
                    .values_list('username', flat=True)
            )

        return self.check_if_changed(
            'tvm_tirole_ids',
            valid, _('Можно указать только TVM-приложения, для которых вы являетесь управляющим в ABC'),
        )

    def clean_plugin_type(self):
        def valid(ptype):
            return ptype in (choice[0] for choice in SYSTEM_PLUGIN_TYPE.UNPRIVILEGED_CHOICES)

        return self.check_if_changed('plugin_type', valid, _('Недостаточно прав для выбора этой системы подключения'))

    def clean_node_plugin_type(self):
        def valid(ptype):
            return ptype in ('', SYSTEM_NODE_PLUGIN_TYPE.YAML)

        error_message = _('Можно указать только YAML или отсутствие плагина')
        value = self.check_if_changed('node_plugin_type', valid, error_message)
        return value if value else None

    def clean(self):
        cleaned_data = super(SoxSystemEditForm, self).clean()

        if self.errors:
            return

        if 'name' in self.changed_data:
            if not cleaned_data.get('name', {}).get('en') or not cleaned_data.get('name', {}).get('ru'):
                raise forms.ValidationError(_('Name can not be empty'))

        def valid_base_url(_):
            change_to_cert = cleaned_data.get('auth_factor') == SYSTEM_AUTH_FACTOR.CERT
            change_to_tvm = cleaned_data.get('auth_factor') == SYSTEM_AUTH_FACTOR.TVM
            initial_tvm = self.system.auth_factor == SYSTEM_AUTH_FACTOR.TVM
            return change_to_tvm or initial_tvm and not change_to_cert

        self.check_if_changed('base_url', valid_base_url, _('base url нельзя редактировать, если методом '
                                                            'аутентификации выбран клиентский сертификат'))

        return cleaned_data


class SystemEditForm(SoxSystemEditForm):
    is_sox = fields.NullBooleanField(required=False, label=_('sox'))
    is_broken = fields.NullBooleanField(required=False, label=_('Is broken'))
    has_review = fields.NullBooleanField(required=False, label=_('has review'))
    review_on_relocate_policy = forms.ChoiceField(label=_('review on relocate policy'), required=False,
                                                  choices=SYSTEM_REVIEW_ON_RELOCATE_POLICY.CHOICES)
    inconsistency_policy = forms.ChoiceField(label=_('inconsistency policy'), required=False,
                                             choices=SYSTEM_INCONSISTENCY_POLICY.CHOICES)
    workflow_approve_policy = forms.ChoiceField(label=_('workflow approve policy'), required=False,
                                                choices=SYSTEM_WORKFLOW_APPROVE_POLICY.CHOICES)

    def clean(self):
        cleaned_data = super(SystemEditForm, self).clean()

        if self.errors:
            return

        if 'is_sox' in self.changed_data and cleaned_data['is_sox']:
            if (
                cleaned_data.get('has_review') is not True or
                cleaned_data.get('review_on_relocate_policy') != SYSTEM_REVIEW_ON_RELOCATE_POLICY.REVIEW or
                cleaned_data.get('inconsistency_policy') != SYSTEM_INCONSISTENCY_POLICY.STRICT_SOX or
                cleaned_data.get('workflow_approve_policy') != SYSTEM_WORKFLOW_APPROVE_POLICY.ANOTHER
            ):
                raise forms.ValidationError(
                    _('has_review, review_on_relocate_policy, inconsistency_policy, workflow_approve_policy should be set to sox defaults')
                )

        return cleaned_data


class ExtendedSystemEditForm(SystemEditForm):
    slug = forms.CharField(label=_('Slug'), required=False)
    roles_review_days = forms.IntegerField(required=False, label=_('roles review days'))
    endpoint_timeout = forms.IntegerField(required=False, label=_('endpoint timeout'))
    endpoint_long_timeout = forms.IntegerField(required=False, label=_('endpoint long timeout'))
    max_approvers = forms.IntegerField(required=False, label=_('max approvers'))
    can_be_broken = fields.NullBooleanField(required=False, label=_('Can be broken'))
    auth_factor = forms.ChoiceField(required=False, label=_('auth factor'), choices=SYSTEM_AUTH_FACTOR.CHOICES)

    def clean_plugin_type(self):
        return self.cleaned_data['plugin_type']


class FavoriteSystemForm(BaseForm):
    system = fields.SystemField(label=_('Система'), required=True)


class BaseRoleForm(BaseForm):
    system = fields.SystemField(label=_('Система'), required=False, select_related=['root_role_node'])
    user = fields.UsersField(enforce_existence=False, label=_('Пользователь'), required=False)
    user_type = forms.ChoiceField(choices=USER_TYPES.CHOICES,
                                  label=_('Тип пользователя – владельца роли'), required=False)
    group = fields.GroupsField(label=_('Группа'), required=False)
    path = fields.RoleNodeValueField(label=_('Путь роли'), required=False)
    nodeset = fields.RoleNodeSetField(label=_('Набор ролей'), required=False)

    def __init__(self, *args, **kwargs):
        super(BaseRoleForm, self).__init__(*args, **kwargs)
        self.fields['path'].form = weakref.proxy(self)
        self.fields['nodeset'].form = weakref.proxy(self)


class RoleForm(BaseRoleForm):
    TYPES = (
        ('active', 'active'),
        ('requested', 'requested'),
        ('inactive', 'inactive'),
        ('returnable', 'returnable'),
    )
    OWNERSHIPS = (
        ('personal', 'Персональная роль'),
        ('group', 'Групповая роль'),
    )
    PARENT_TYPES = (
        ('absent', 'Роль без родительской'),
        ('present', 'Роль с родительской'),
        ('user', 'Роль с родительской ролью, принадлежащей пользователю'),
        ('group', 'Роль с родительской ролью, принадлежащей группе'),
    )
    id = fields.CommaSeparatedIntegerField(label=_('Ids'), required=False)
    id__gt = forms.IntegerField(min_value=0, label=_('Id >'), required=False)
    id__lt = forms.IntegerField(min_value=1, label=_('Id <'), required=False)
    sox = fields.NullBooleanField(label=_('SOX'), required=False)
    users = fields.UsersField(enforce_existence=False, label=_('Пользователь'), required=False)
    type = forms.ChoiceField(label=_('Тип'), choices=TYPES, required=False)
    ownership = forms.ChoiceField(label=_('Фильтр по персональным/групповым ролям'), required=False,
                                  choices=OWNERSHIPS)
    parent = fields.RoleField(label=_('Родительская роль'), required=False)
    parent_type = forms.ChoiceField(label=_('Тип родительской роли'), required=False, choices=PARENT_TYPES)
    state = fields.RoleStateField(required=False)
    internal_role = forms.ChoiceField(choices=[(role[0], role[1]['name']['ru'])
                                               for role in settings.IDM_SYSTEM_ROLES],
                                      label=_('Внутренняя роль'), required=False)
    fields_data = fields.MaybeJSONField(required=False, label=_('Данные полей'))
    field_data = fields.LookupCharField(label=_('Фильтр по полям'), required=False)
    role__contains = forms.CharField(label=_('Ключевые слова'), required=False)
    with_parents = fields.NullBooleanField(label=_('Проверить всех в иерархии'), required=False)
    abc_slug = forms.CharField(label='Слаг ABC сервиса', required=False)


class RoleDepriveForm(BaseForm):
    validate_depriving = forms.BooleanField(required=False)
    comment = forms.CharField(required=False, label=_('Комментарий'))


class ApproverForm(BaseForm):
    approver = fields.UsersField(label=_('Пользователь'), required=False)


class AllApproversForm(BaseForm):
    approve_id = forms.ModelChoiceField(queryset=Approve.objects.all(), label=_('Подтверждающий'))


class ApproveRequestForm(BaseRoleForm):
    STATUS_PENDING = 'pending'
    STATUS_PROCESSED = 'processed'
    STATUS_ALL = 'all'
    STATUSES = (
        (STATUS_PENDING, _('Необработанные')),
        (STATUS_PROCESSED, _('Обработанные')),
        (STATUS_ALL, _('Все')),
    )

    approver = fields.UsersField(label=_('Пользователь'), required=False)
    status = forms.ChoiceField(
        label=_('Состояние запроса'),
        choices=STATUSES,
        initial='pending',
    )
    priority_type = forms.ChoiceField(
        label=_('Тип приоритета'),
        choices=PriorityType.choices(),
        initial='primary',
        required=False
    )
    role_ids = fields.CommaSeparatedIntegerField(label=_('ID ролей'), required=False)
    reason = forms.ChoiceField(
        label=_('Причина запроса'),
        choices=RoleRequest.REQUEST_REASONS,
        initial='unknown',
        required=False,
    )
    requester = fields.UsersField(label=_('Пользователи'), required=False)


class ApproveRequestBulkForm(BaseForm):
    user = fields.UserField(label=_('Пользователь'), required=False)
    system = fields.SystemField(label=_('Система'), required=False)
    path = fields.RoleNodeValueField(label=_('Данные роли'), required=False)
    role_ids = fields.CommaSeparatedIntegerField(label=_('ID ролей'), required=False)
    decision = forms.ChoiceField(choices=ApproveRequest.DECISIONS, required=False)
    comment = forms.CharField(required=False, label=_('Комментарий'))
    priority_type = forms.ChoiceField(
        label=_('Тип приоритета'),
        choices=PriorityType.choices(),
        initial='primary',
    )

    def __init__(self, *args, **kwargs):
        super(ApproveRequestBulkForm, self).__init__(*args, **kwargs)
        self.fields['path'].form = weakref.proxy(self)


class RoleRequestFilterForm(BaseForm):
    system = fields.SystemField(label=_('Система'), required=False)
    requester = fields.UserField(label=_('Запросивший'), required=False)
    path = fields.RoleNodeValueField(label=_('Данные роли'), required=False)
    is_done = forms.BooleanField(label=_('Завершено'), required=False)


class RoleRequestForm(SubjectFormMixin, BaseForm):
    # флаг, показывающий, что нужно просто вернуть аппруверов для такого запроса
    simulate = forms.BooleanField(required=False, initial=False)
    system = fields.SystemField(label=_('Система'), select_related=['actual_workflow'])
    path = fields.RoleNodeValueField(label=_('Данные роли'))
    fields_data = fields.NoValidationField(required=False, label=_('Данные полей'))
    request_fields = fields.JSONField(required=False, label=_('Данные запроса'))
    comment = forms.CharField(required=False, label=_('Комментарий'))
    silent = forms.BooleanField(required=False, label=_('Не отправлять письмо о получении роли'))
    deprive_after_days = forms.IntegerField(
        label=_('Время жизни роли в днях'), required=False, min_value=1, max_value=MAX_ROLE_TTL_DAYS
    )
    deprive_at = forms.DateField(input_formats=["%Y-%m-%d"], label=_('Дата истечения срока действия'), required=False)
    review_at = forms.DateField(input_formats=["%Y-%m-%d"], label=_('Дата пересмотра роли'), required=False)
    with_inheritance = fields.NullBooleanTrueField(label=_('Будет ли роль выдана вложенным департаментам'),
                                                   required=False)
    with_robots = fields.NullBooleanTrueField(label=_('Будет ли роль выдана роботам'), required=False)
    with_external = fields.NullBooleanTrueField(label=_('Будет ли роль выдана внешним '), required=False)
    without_hold = forms.BooleanField(label=_('Будут ли персональные роли мгновенно отозваны при выходе из группы'),
                                      required=False)
    no_meta = forms.BooleanField(required=False, initial=False)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields['path'].form = weakref.proxy(self)

    def clean_deprive_at(self):
        value = self.cleaned_data.get('deprive_at')
        field = prepare_date_field(value, _('Нельзя запросить временную роль на дату в прошлом'))
        if field and field > timezone.now() + timezone.timedelta(days=MAX_ROLE_TTL_DAYS):
            raise forms.ValidationError(_('Нельзя запросить временную роль больше чем на год'))

        return field

    def clean_review_at(self):
        value = self.cleaned_data.get('review_at')
        return prepare_date_field(value, _('Нельзя назначить пересмотр на дату в прошлом'))

    def clean(self):
        cleaned_data = super().clean()

        if self.errors:
            return

        if cleaned_data['deprive_after_days'] and cleaned_data['deprive_at']:
            raise forms.ValidationError('Нельзя передавать одновременно время жизни роли в днях и дату истечения')

        return cleaned_data


class RoleRerequestForm(BaseForm):
    REREQUEST_CHOICES = [item for item in ROLE_STATE.STATE_CHOICES.items() if item[0] in ROLE_STATE.REQUESTED_STATES]
    state = forms.ChoiceField(choices=REREQUEST_CHOICES, label='Состояние', error_messages={
        'invalid_choice': 'Невозможно перевести роль в статус "%(value)s"'
    })
    comment = forms.CharField(required=False)
    ttl_days = forms.IntegerField(required=False, min_value=1, max_value=MAX_ROLE_TTL_DAYS)
    ttl_date = forms.DateTimeField(required=False)

    def clean_ttl_date(self):
        value = self.cleaned_data.get('ttl_date')
        field = prepare_date_field(value, _('Нельзя запросить временную роль на дату в прошлом'))
        if field and field > timezone.now() + timezone.timedelta(days=MAX_ROLE_TTL_DAYS):
            raise forms.ValidationError(_('Нельзя запросить временную роль больше чем на год'))

        return field

    def clean(self):
        cleaned_data = super(RoleRerequestForm, self).clean()

        if self.errors:
            return

        if cleaned_data['ttl_days'] and cleaned_data['ttl_date']:
            raise forms.ValidationError('Нельзя передавать одновременно время жизни роли в днях и дату истечения')

        return cleaned_data


class RoleActionForm(BaseForm):
    action = forms.ChoiceField(required=False, label=_('Действие'),
                               choices=((action, ACTION.ACTIONS[action][2])
                                        for action in ACTION.ADDITIONAL_ROLE_API_ACTIONS))


class RoleFieldsRequestForm(SubjectFormMixin, BaseForm):
    system = fields.SystemField(label=_('Система'))
    path = fields.RoleNodeValueField(label=_('Роль'))
    fields_data = fields.NoValidationField(required=False, label=_('Данные полей'))

    def __init__(self, *args, **kwargs):
        super(RoleFieldsRequestForm, self).__init__(*args, **kwargs)
        self.fields['path'].form = weakref.proxy(self)


class RoleAdditionalFieldsRequestForm(SubjectFormMixin, BaseForm):
    system = fields.SystemField(label=_('Система'))


class BaseWorkflowTestForm(BaseForm):
    requester = fields.UserField(error_messages={'invalid_choice': _('Неизвестный запрашивающий роль.')})
    path = fields.RoleNodeValueField()
    request_type = forms.ChoiceField(
        label=_('Тип запроса'), required=False, choices=REQUEST_TYPE.CHOICES, initial=REQUEST_TYPE.REQUEST
    )

    def __init__(self, system, *args, **kwargs):
        super(BaseWorkflowTestForm, self).__init__(*args, **kwargs)
        self.fields['path'].system = system
        self.fields['path'].form = weakref.proxy(self)

    def clean(self):
        if self.errors:
            return

        self.cleaned_data['subject_field'] = self.subject_field
        self.cleaned_data['subject'] = self.cleaned_data[self.subject_field]
        if not self.cleaned_data['request_type']:
            self.cleaned_data['request_type'] = self.fields['request_type'].initial

        return self.cleaned_data


class RoleReportForm(BaseForm):
    user = fields.UserField(label=_('Пользователь'), required=False)
    system = fields.SystemField(label=_('Система'), required=False)
    states = fields.RoleStateField(required=False)
    date_from = forms.DateField(label=_('От'), required=False)
    date_to = forms.DateField(label=_('До'), required=False)


class UserWorkflowTestForm(BaseWorkflowTestForm):
    subject_field = 'user'
    user = fields.UserField(
        error_messages={'invalid_choice': _('Неизвестный пользователь.')},
        label=_('Пользователь'),
    )

    def __init__(self, system, user_type, *args, **kwargs):
        super(UserWorkflowTestForm, self).__init__(system, *args, **kwargs)

        user_queryset = self.fields['user'].queryset
        self.fields['user'].queryset = user_queryset.filter(type=user_type)


class GroupWorkflowTestForm(BaseWorkflowTestForm):
    subject_field = 'group'
    group = fields.GroupField(label=_('Группа'))


class RoleApproveForm(forms.Form):
    """Форма подтверждения роли
    """
    approve = forms.NullBooleanField(required=False)
    decision = forms.ChoiceField(choices=ApproveRequest.DECISIONS, required=False)
    comment = forms.CharField(required=False)

    def clean(self):
        approve = self.cleaned_data.get('approve')
        decision = self.cleaned_data.get('decision')

        if (approve is None and not decision):
            self.add_error('decision', 'approve or decision should be specified')
        elif not decision:
            if approve is True:
                decision = APPROVEREQUEST_DECISION.APPROVE
            else:
                decision = APPROVEREQUEST_DECISION.DECLINE

        self.cleaned_data['decision'] = decision
        return self.cleaned_data


class ActionForm(BaseForm):
    id = forms.IntegerField(label=_('Id'), required=False)
    user = fields.UsersField(enforce_existence=False, label=_('Пользователи'), required=False)
    user_type = forms.ChoiceField(choices=USER_TYPES.CHOICES,
                                  label=_('Тип пользователя – владельца роли'), required=False)
    group = fields.GroupsField(label=_('Группа'), required=False)
    role = fields.RoleField(label=_('Роль'), required=False, select_related=['system'])
    system = fields.SystemField(label=_('Система'), required=False)
    path = fields.RoleNodeValueField(label=_('Путь роли'), required=False)
    action = fields.ActionTypesField(label=_('Тип действия'), required=False)
    is_active = fields.NullBooleanField(label=_('Активно'), required=False)
    date_from = forms.DateField(label=_('От'), required=False)
    date_to = forms.DateField(label=_('До'), required=False)

    def __init__(self, *args, **kwargs):
        super(ActionForm, self).__init__(*args, **kwargs)
        self.fields['path'].form = weakref.proxy(self)


class ReportPostForm(BaseForm):
    format = forms.ChoiceField(label=_('Формат'), choices=(('csv', 'CSV'), ('xls', 'XLS')))
    comment = forms.CharField(label=_('Комментарий'), required=False)


class GroupChangedRolesForm(BaseForm):
    group = fields.GroupField(label=_('Группа для изменения'), required=True)
    parent = fields.GroupField(label=_('Новый родитель'), required=False)
    membership_inheritance = fields.NullBooleanField(label=_('Новое значение галки'), required=False)

    def clean_group(self):
        group = self.cleaned_data['group']
        if group.type not in [GROUP_TYPES.SERVICE, GROUP_TYPES.DEPARTMENT]:
            self.add_error('group', 'works only with services and departments')

        return group

    def clean(self):
        if self.errors:
            return
        group = self.cleaned_data.get('group')
        parent = self.cleaned_data.get('parent')
        membership_inheritance = self.cleaned_data.get('membership_inheritance')

        if (parent is None) == (membership_inheritance is None):
            self.add_error(None, 'exactly one of fields parent or membership_inheritance should be defined')
        if membership_inheritance is not None and group.type != GROUP_TYPES.SERVICE:
            self.add_error('group', 'only services has membership_inheritance')

        return self.cleaned_data


class PassportLoginAssignmentForm(BaseForm):
    passport_login = fields.PassportLoginField(required=True)

    def has_changed(self):
        # При передаче пустой строки '' метод суперкласса завершает обработку формы
        # нам же пустое значение нужно для отвязки логина
        return True


class MembershipForm(BaseForm):
    MODES = (
        ('direct', _('Непосредственные')),
        ('all', _('Все'))
    )
    DIRECTIONS = (
        ('up', _('Группы вверх по иерархии (страница пользователя)')),
        ('down', _('Группы вниз по иерархии (страница группы)'))
    )
    group = fields.GroupsField(label=_('Группа'), required=False)
    user = fields.UsersField(label=_('Пользователь'), required=False)
    is_active = fields.NullBooleanTrueField(label=_('Активно'), required=False)
    mode = forms.ChoiceField(label=_('Режим отображения членств'), choices=MODES, required=False)
    direction = forms.ChoiceField(label=_('Направление просмотра групп'), choices=DIRECTIONS, required=False)
    group__type = fields.CommaSeparatedMultipleChoiceField(label=_('Тип группы'), choices=Group.GROUP_CHOICES,
                                                           required=False)
    state = forms.ChoiceField(label=_('Статус членства'), choices=GROUPMEMBERSHIP_STATE.NAMES, required=False)

    def clean(self):
        if not self.cleaned_data.get('user') and not self.cleaned_data.get('group'):
            raise forms.ValidationError(_('Укажите, пожалуйста, пользователя или группу'))
        return self.cleaned_data


class RoleRequestPermissionsForm(SubjectFormMixin, BaseForm):
    system = fields.SystemField(label=_('Система'), required=True)
    path = fields.RoleNodeValueField(label=_('Данные роли'), required=False)
    fields_data = fields.NoValidationField(label=_('Данные полей'), required=False)

    def __init__(self, *args, **kwargs):
        super(RoleRequestPermissionsForm, self).__init__(*args, **kwargs)

        self.fields['path'].form = weakref.proxy(self)


class BatchResourceForm(BaseForm):
    id = forms.CharField(required=False)
    method = forms.CharField()
    path = forms.CharField()
    body = forms.Field(required=False)


class PermissionsForm(BaseForm):
    system = fields.SystemField(label=_('Система'), required=False)
    path = fields.RoleNodeValueField(label=_('Путь'), required=False)

    def __init__(self, *args, **kwargs):
        super(PermissionsForm, self).__init__(*args, **kwargs)
        self.fields['path'].form = weakref.proxy(self)


class InconsistencyForm(BaseForm):
    state = fields.CommaSeparatedMultipleChoiceField(choices=Inconsistency.STATE_CHOICES, required=False)
    type = fields.CommaSeparatedMultipleChoiceField(choices=Inconsistency.TYPES, required=False)
    system = fields.SystemField(label=_('Система'), required=False)
    our_role = fields.RoleField(label=_('Роль, из-за которой заведено это расхождение'), required=False)
    new_role = fields.RoleField(label=_('Роль, заведённая по этому расхождению'), required=False)
    user = fields.CommaSeparatedCharField(required=False)
    user_type = forms.ChoiceField(choices=USER_TYPES.CHOICES,
                                  label=_('Тип пользователя – владельца роли'), required=False)
    group = fields.CommaSeparatedIntegerField(required=False)
    subject = fields.CommaSeparatedCharField(required=False)
    role = fields.RoleNodeValueField(required=False)

    def __init__(self, *args, **kwargs):
        super(InconsistencyForm, self).__init__(*args, **kwargs)
        self.fields['role'].form = weakref.proxy(self)

    def clean(self):
        value = self.cleaned_data.get('subject')
        if not value:
            return self.cleaned_data

        self.cleaned_data['user'] = [str(x) for x in value]

        group_ids = []
        for item in value:
            try:
                group_id = int(item)
                group_ids.append(group_id)
            except (ValueError, TypeError):
                pass
        self.cleaned_data['group'] = group_ids
        return self.cleaned_data


class SystemOperationForm(BaseForm):
    DOESNOT_SUPPORT_DRYRUN = ('handle', 'recover')
    OPERATIONS = (
        ('pull_handle', _('Запрос к ручке системы')),
        ('sync_nodes', _('Синхронизация узлов дерева ролей')),
        ('sync_roles', _('Синхронизация ролей')),
        ('sync_memberships', _('Синхронизация членств в группах')),
        ('recover', _('Включение')),
    )
    operation = forms.ChoiceField(label=_('Операция'), choices=OPERATIONS)
    dry_run = forms.BooleanField(label=_('Подробный ответ'), initial=False, required=False)
    options = fields.NoValidationField(required=False, label=_('Параметры'))
    resolve_in_idm_direct = forms.BooleanField(label=_('Разрешать в пользу IDM'), initial=False, required=False)

    def clean(self):
        if self.errors:
            return

        if self.cleaned_data['dry_run'] and self.cleaned_data['operation'] in self.DOESNOT_SUPPORT_DRYRUN:
            # этот параметр не поддерживается, проверка нужна для подстраховки от ошибки клиента
            raise forms.ValidationError(_('Эта операция не поддерживает режим dry_run'))
        if self.cleaned_data['resolve_in_idm_direct'] and self.cleaned_data['operation'] != 'sync_roles':
            raise forms.ValidationError(_('Эта операция не поддерживает параметр resolve_in_idm_direct'))

        return self.cleaned_data


class SystemHandleForm(forms.Form):
    handle = forms.ChoiceField(label=_('Ручка'), choices=HANDLE_CHOICES, required=True)
    auth_factor = forms.ChoiceField(label=_('Фактор авторизации'), choices=SYSTEM_AUTH_FACTOR.CHOICES, required=True)
    library = forms.ChoiceField(label=_('Библиотека'), choices=LIBRARY_CHOICES, required=True, initial='default')


class UserForm(BaseForm):
    updated__since = forms.DateTimeField(label=_('От'), required=False)
    updated__until = forms.DateTimeField(label=_('До'), required=False)
    is_active = fields.NullBooleanField(label=_('Активность'), required=False)


class UserUpdateForm(BaseForm):
    notify_responsibles = forms.BooleanField(
        label=_('Уведомлять ответственных о событиях по ролям'),
        required=False
    )
    is_frozen = forms.BooleanField(
        label=_('Признак замороженности ролей пользователя'),
        required=False
    )


class GroupForm(BaseForm):
    updated__since = forms.DateTimeField(label=_('От'), required=False)
    updated__until = forms.DateTimeField(label=_('До'), required=False)
    is_active = fields.NullBooleanField(label=_('Активность'), required=False)
    type = forms.ChoiceField(label=_('Тип'), choices=Group.GROUP_CHOICES, required=False)


class RoleNodeForm(BaseForm):
    system = fields.SystemField(label=_('Система'), required=False)
    is_key = fields.NullBooleanField(label=_('Только key- или value- узлы'), required=False)
    is_public = fields.NullBooleanField(label=_('Только public узлы'), required=False)
    is_exclusive = fields.NullBooleanField(label=_('Эксклюзивная'), required=False)
    slug_path = fields.RoleNodeSlugField(label=_('Данные роли'), required=False)
    parent = fields.RoleNodeSlugField(label=_('Предок'), required=False)
    updated__since = fields.TZAwareDateTimeField(label=_('От'), required=False)
    updated__until = fields.TZAwareDateTimeField(label=_('До'), required=False)
    state = forms.CharField(label=_('Состояние'), required=False)

    def __init__(self, *args, **kwargs):
        super(RoleNodeForm, self).__init__(*args, **kwargs)
        self.fields['slug_path'].form = weakref.proxy(self)
        self.fields['parent'].form = weakref.proxy(self)


class AliasForm(BaseForm):
    type = forms.CharField(required=False, label=_('Тип'))
    name = fields.LangField(required=True, label=_('Название'))

    def clean_type(self):
        return self.cleaned_data.get('type') or RoleAlias.DEFAULT_ALIAS

    def clean(self):
        if 'name' in self.cleaned_data:
            name_ru, name_en = get_lang_pair(self.cleaned_data['name'])
            self.cleaned_data['name'] = name_ru
            self.cleaned_data['name_en'] = name_en
        return self.cleaned_data


class FieldForm(BaseForm):
    slug = forms.CharField(required=True, label=_('Slug'))
    type = forms.CharField(required=False, label=_('Тип'))
    required = forms.BooleanField(required=False, label=_('Обязательное'))
    name = fields.LangField(required=True, label=_('Название'))
    options = fields.NoValidationField(required=False, label=_('Опции'))
    dependencies = fields.NoValidationField(required=False, label=_('Зависимости'))

    def clean_type(self):
        value = self.cleaned_data.get('type') or FIELD_TYPE.CHARFIELD
        return value

    def clean_dependencies(self):
        value = self.cleaned_data.get('dependencies')
        if value is None or value == '':
            return None
        if not isinstance(value, dict):
            raise forms.ValidationError(_('Зависимости должны быть словарём'))
        return value

    def _clean_options_after_type(self, field_type):
        value = self.cleaned_data.get('options')
        if value == '':
            value = None
        try:
            RoleField.check_options(field_type, value)
        except ValueError as e:
            raise forms.ValidationError(str(e))
        return value

    def clean(self):
        self.cleaned_data['options'] = self._clean_options_after_type(self.cleaned_data.get('type'))
        name_ru, name_en = get_lang_pair(self.cleaned_data['name'])
        self.cleaned_data['name'] = name_ru
        self.cleaned_data['name_en'] = name_en
        if 'required' in self.cleaned_data:
            self.cleaned_data['is_required'] = self.cleaned_data.pop('required')
        return self.cleaned_data


class ResponsibilityForm(BaseForm):
    username = forms.CharField(required=True, label=_('Сотрудник'))
    notify = forms.BooleanField(required=False, label=_('Уведомлять'))

    def clean(self):
        if 'username' in self.cleaned_data:
            self.cleaned_data['user'] = self.cleaned_data.pop('username')
        return self.cleaned_data


class BaseUpdateRoleNodeForm(BaseForm):
    name = fields.LangField(label=_('Название'), required=False)
    help = fields.LangField(label=_('Описание'), required=False)
    visibility = fields.NullBooleanField(label=_('Видимость'), required=False)
    is_exclusive = fields.NullBooleanField(label=_('Эксклюзивная'), required=False)
    set = forms.CharField(label=_('Группа'), required=False, max_length=100)
    unique_id = fields.NullableCharField(label=_('Уникальный id'), required=False, max_length=255)
    review_required = fields.NullBooleanField(label=_('Требуется ревью'), required=False)
    comment_required = forms.BooleanField(label=_('Требуется комментарий при запросе роли'), required=False)
    node_aliases = fields.NestedObjectField(AliasForm, label=_('Алиасы'), required=False)
    node_fields = fields.NestedObjectField(FieldForm, label=_('Поля'), required=False)
    node_responsibilities = fields.NestedObjectField(ResponsibilityForm, label=_('Ответственные'), required=False)

    instance: RoleNode

    def clean(self):
        self.cleaned_data = super().clean()
        if self.cleaned_data.get('node_responsibilities'):
            usernames = (x['user'] for x in self.cleaned_data['node_responsibilities'])
            users = {x.username: x for x in User.objects.filter(username__in=usernames)}
            for responsibility in self.cleaned_data['node_responsibilities']:
                user = users.get(responsibility['user'])
                if user is None:
                    raise forms.ValidationError(
                        _('Пользователь %(value)s не найден'),
                        code='invalid_choice',
                        params={'value': responsibility['user']},
                    )
                responsibility['user'] = user

        return self.cleaned_data


class UpdateRoleNodeForm(BaseUpdateRoleNodeForm):
    parent = fields.RoleNodeSlugField(label=_('Родитель'), required=False)
    slug = forms.CharField(label=_('Slug'), required=False)
    create = forms.BooleanField(label=_('Создавать отсутствующие узлы'), required=False)

    def __init__(self, *args, **kwargs):
        self.instance = kwargs.pop('instance', None)
        self.system = kwargs.pop('system')
        self.derived_slug = kwargs.pop('slug')  # slug, который мы достаём из url
        self.derived_parent = kwargs.pop('parent')  # parent, который мы достаём из url
        super(UpdateRoleNodeForm, self).__init__(*args, **kwargs)

        self.fields['parent'].system = self.system

    def clean_unique_id(self):
        unique_id = self.cleaned_data['unique_id']
        qs = RoleNode.objects.active().filter(system=self.system, unique_id=unique_id)
        if self.instance:
            qs = qs.exclude(pk=self.instance.pk)
        if unique_id and qs.exists():
            raise forms.ValidationError(_('В данной системе уже есть такой unique_id: %s.') % unique_id)
        return unique_id

    def clean(self):
        data = super().clean()

        effective_parent = data.get('parent') or self.derived_parent
        if effective_parent.roles.returnable().exists():
            raise forms.ValidationError('Parent node already has active or soon-to-be active roles')

        if self.instance:
            if effective_parent.is_key == self.instance.is_key:
                raise forms.ValidationError(_('Невозможно сменить родителя узла на указанного. '
                                              'Ключевые и неключевые узлы должны чередоваться.'))

            if effective_parent.is_descendant_of(self.instance):
                raise forms.ValidationError(_('Невозможно перенести узел в свое же поддерево.'))

        base_path = effective_parent.slug_path
        slug = data.get('slug') or self.derived_slug
        role_path = '{}{}/'.format(base_path, slug)

        nodes_qset = self.system.nodes.active().filter(slug_path=role_path)
        if self.instance:
            nodes_qset = nodes_qset.exclude(pk=self.instance.pk)
        if nodes_qset.exists():
            raise forms.ValidationError('Узел "%s" уже есть на этом уровне дерева' % role_path)

        return data

    def as_canonical(self):
        if self.instance:
            canonical = self.instance.as_canonical()
        else:
            canonical = CanonicalNode(slug=self.cleaned_data['slug'], hash='')

        slug = self.cleaned_data.get('slug')
        if not slug:
            slug = canonical.slug
        name = self.cleaned_data.get('name')
        if name is not None:
            name_ru, name_en = get_lang_pair(name)
        else:
            name_ru, name_en = canonical.name, canonical.name_en

        help = self.cleaned_data.get('help')
        if help is not None:
            help_ru, help_en = get_lang_pair(self.cleaned_data.get('help'))
        else:
            help_ru, help_en = canonical.description, canonical.description_en

        is_public = self.cleaned_data.get('visibility', True)
        if is_public is None:
            is_public = canonical.is_public

        is_exclusive = self.cleaned_data.get('is_exclusive')
        if is_exclusive is None:
            is_exclusive = canonical.is_exclusive

        unique_id = self.cleaned_data.get('unique_id')
        if unique_id is None:
            unique_id = canonical.unique_id

        review_required = canonical.review_required
        if 'review_required' in self.cleaned_data:
            review_required = self.cleaned_data['review_required']
        comment_required = self.cleaned_data.get('comment_required', canonical.review_required)

        set_ = self.cleaned_data.get('set')
        if set_ is None:
            set_ = canonical.set

        node_aliases = self.cleaned_data.get('node_aliases')
        if node_aliases is not None:
            aliases = [CanonicalAlias(**item) for item in node_aliases]
        else:
            aliases = canonical.aliases

        node_fields = self.cleaned_data.get('node_fields')
        if node_fields is not None:
            fields_ = [CanonicalField(**item) for item in node_fields]
        else:
            fields_ = canonical.fields

        node_responsibilities = self.cleaned_data.get('node_responsibilities')
        if node_responsibilities is not None:
            responsibilities = [CanonicalResponsibility(**item) for item in node_responsibilities]
        else:
            responsibilities = canonical.responsibilities

        modified_canonical = CanonicalNode(
            slug=slug,
            name=name_ru,
            name_en=name_en,
            description=help_ru,
            description_en=help_en,
            is_public=is_public,
            is_exclusive=is_exclusive,
            unique_id=unique_id,
            review_required=review_required,
            comment_required=comment_required,
            set=set_,
            aliases={alias.as_key(): alias for alias in aliases},
            fields={field.as_key(): field for field in fields_},
            responsibilities={responsibility.as_key(): responsibility for responsibility in responsibilities},

            children=(),
            hash='',
        )
        return modified_canonical


class CreateRoleNodeForm(BaseUpdateRoleNodeForm):
    system = fields.SystemField(label=_('Система'), required=True)
    slug = forms.CharField(label=_('Slug'), required=True)
    name = fields.LangField(label=_('Название'), required=True)
    parent = fields.RoleNodeSlugField(label=_('Данные роли'), required=True)

    def __init__(self, *args, **kwargs):
        super(CreateRoleNodeForm, self).__init__(*args, **kwargs)
        self.fields['parent'].form = weakref.proxy(self)

    def clean(self):
        if self.errors:
            return

        cleaned_data = super(CreateRoleNodeForm, self).clean()
        role_path = '%s%s/' % (cleaned_data['parent'].slug_path, cleaned_data['slug'])
        if cleaned_data['system'].nodes.active().filter(slug_path=role_path).exists():
            raise forms.ValidationError('Узел "%s" уже есть на этом уровне дерева' % role_path)

        if cleaned_data['parent'].is_value and cleaned_data['parent'].get_children().filter(
            state__in=RoleNode.ACTIVE_STATES
        ).exists():
            raise forms.ValidationError('У родительского узла уже есть потомок, больше добавлять нельзя')

        if cleaned_data['parent'].roles.returnable().exists():
            raise forms.ValidationError('Parent node already has active or soon-to-be active roles')

        return cleaned_data

    def as_canonical(self):
        if self.cleaned_data.get('name') is not None:
            name_ru, name_en = get_lang_pair(self.cleaned_data.get('name'))
        else:
            name_ru, name_en = '', ''

        unique_id = self.cleaned_data.get('unique_id')
        if unique_id is None:
            unique_id = ''

        review_required = self.cleaned_data.get('review_required')
        comment_required = self.cleaned_data.get('comment_required', False)

        if self.cleaned_data.get('help') is not None:
            help_ru, help_en = get_lang_pair(self.cleaned_data.get('help'))
        else:
            help_ru, help_en = '', ''
        is_public = self.cleaned_data.get('visibility', True)
        if is_public is None:
            is_public = True

        is_exclusive = self.cleaned_data.get('is_exclusive')
        if is_exclusive is None:
            is_exclusive = False

        node_aliases = self.cleaned_data.get('node_aliases')
        if node_aliases is not None:
            aliases = [CanonicalAlias(**item) for item in node_aliases]
        else:
            aliases = []

        node_fields = self.cleaned_data.get('node_fields')
        if node_fields is not None:
            fields_ = [CanonicalField(**item) for item in node_fields]
        else:
            fields_ = []

        node_responsibilities = self.cleaned_data.get('node_responsibilities')
        if node_responsibilities is not None:
            responsibilities = [CanonicalResponsibility(**item) for item in node_responsibilities]
        else:
            responsibilities = []

        canonical = CanonicalNode(
            slug=self.cleaned_data['slug'],
            name=name_ru,
            name_en=name_en,
            description=help_ru,
            description_en=help_en,
            is_public=is_public,
            is_exclusive=is_exclusive,
            unique_id=unique_id,
            review_required=review_required,
            comment_required=comment_required,
            set=self.cleaned_data.get('set', ''),
            aliases={alias.as_key(): alias for alias in aliases},
            fields={field.as_key(): field for field in fields_},
            responsibilities={responsibility.as_key(): responsibility for responsibility in responsibilities},

            children=(),
            hash='',
        )
        return canonical


class TransferForm(BaseForm):
    user = fields.UsersField(label=_('Пользователи'), required=False)
    group = fields.GroupsField(label=_('Группы'), required=False)
    type = fields.CommaSeparatedMultipleChoiceField(label=_('Типы'), required=False, choices=Transfer.TYPE_CHOICES)
    state = forms.ChoiceField(label=_('Состояния'), required=False, choices=Transfer.STATE_CHOICES)


class TransferAcceptForm(BaseForm):
    """Форма подтверждения перемещения"""

    accept = forms.BooleanField(required=False)


class IdValidationForm(forms.Form):
    """Форма проверки фалидности ключа для пагинатора"""
    query = forms.ModelChoiceField(queryset=None, to_field_name='')

    def __init__(self, objects, key, *args, **kwargs):
        super(IdValidationForm, self).__init__(*args, **kwargs)
        self.fields['query'].queryset = objects
        self.fields['query'].to_field_name = key
