# coding: utf-8


from django.db.models import Q
from tastypie import fields

from idm.api.exceptions import BadRequest
from idm.api.frontend import forms
from idm.api.frontend.base import FrontendApiResource
from idm.users import ranks
from idm.users.models import Group


class GroupResource(FrontendApiResource):
    """
    Ресурс группы
    """

    description = fields.CharField(use_in='detail', attribute='description')
    deprived_at = fields.DateTimeField(use_in='detail', attribute='expire_at', null=True)

    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = Group
        # parent нужен, чтобы получить `url` у скоупа
        queryset = Group.objects.user_groups().nonroot().select_related('parent')
        detail_uri_name = 'external_id'
        resource_name = 'groups'
        detail_allowed_methods = ['get']
        list_allowed_methods = ['get']
        fields = ['created_at', 'slug', 'external_id', 'state', 'name', 'description', 'type',
                  'updated_at', 'deprived_at']
        limit = 100

    def dehydrate(self, bundle):
        bundle = super(GroupResource, self).dehydrate(bundle)

        bundle.data['id'] = bundle.data.pop('external_id')
        bundle.data['url'] = bundle.obj.get_external_url()
        bundle.data['name'] = bundle.obj.get_name()

        return bundle

    def dehydrate_for_detail(self, bundle):
        # Иерархия (сначала корень)
        qs_ancestors = (bundle.obj.get_ancestors(include_self=False)
                        .exclude(level=0).order_by('level'))
        bundle.data['ancestors'] = [
            {
                'id': group.external_id,
                'name': group.name,
                'url': group.slug,
            }
            for group in qs_ancestors
        ]

        # Ответственные
        bundle.data['responsibles'] = [
            {
                'full_name': user.get_full_name(),
                'username': user.username,
            }
            for user in bundle.obj.get_responsibles(ranks=ranks.ALL_HEAD_TYPES)
        ]

        return bundle

    def apply_filters(self, request, applicable_filters, **kwargs):
        return self.get_object_list(request).filter(applicable_filters)

    def build_filters(self, request, filters=None):
        form = forms.GroupForm(filters)
        if not form.is_valid():
            raise BadRequest(form.errors)

        query = form.cleaned_data
        kwargs = {}

        if query['is_active'] is not None:
            if query['is_active']:
                kwargs['state__in'] = ('active',)
            else:
                kwargs['state__in'] = ('deprived', 'depriving')

        if query['type'] != '':
            kwargs['type'] = query['type']

        if query['updated__since'] is not None:
            kwargs['updated_at__gte'] = query['updated__since']

        if query['updated__until'] is not None:
            kwargs['updated_at__lte'] = query['updated__until']

        return Q(**kwargs)
