from contextlib import closing

from django.conf import settings
from django.db import connection
from django.db.models import QuerySet
from django.db.models.query import EmptyResultSet
from django.db.models.expressions import OrderBy, F
from django.utils import six
from django.utils.encoding import force_text
from django.utils.translation import ugettext as _
from psycopg2.sql import Literal
from tastypie.paginator import Paginator
from urllib.parse import parse_qsl
from idm.api.exceptions import BadRequest
from idm.api.frontend.forms import IdValidationForm
from idm.utils.attributes import get_attribute

from urllib.parse import urlencode, splitquery


__all__ = ('EstimatingPaginator', 'OptimizedPaginator', 'DummyPaginator')


class DummyPaginator(Paginator):
    def page(self):
        return {self.collection_name: self.objects}


class EstimatingPaginator(Paginator):
    def __init__(self, *args, use_estimate=True, use_random_sorting_in_unwrap=False, nested_values=None, **kwargs):
        self.use_estimate = use_estimate
        self.use_random_sorting_in_unwrap = use_random_sorting_in_unwrap
        super(EstimatingPaginator, self).__init__(*args, **kwargs)
        self.count_is_estimated = None
        self.nested_values = nested_values

    def get_limit(self):
        limit = self.request_data.get('limit', self.limit)
        if limit in (0, '0'):
            return 0
        return super(EstimatingPaginator, self).get_limit()

    def get_slice(self, limit, offset):
        if limit == 0:
            return []
        slice_ = super(EstimatingPaginator, self).get_slice(limit, offset)
        if hasattr(slice_, 'unwrap'):
            slice_ = slice_.unwrap(use_random_sorting=self.use_random_sorting_in_unwrap)
            if self.nested_values:
                slice_ = slice_.nested_values(*self.nested_values)

        return slice_

    def could_use_estimate(self):
        return (
            self.use_estimate and
            self.resource_uri.startswith('/api/frontend/') and
            connection.vendor == 'postgresql' and
            isinstance(self.objects, QuerySet) and
            not self.objects.query.is_empty()
        )

    def get_estimated_count(self):
        try:
            sql, params = self.objects.values_list('id').query.sql_with_params()
        except EmptyResultSet:
            return 0, False
        with closing(connection.cursor()) as cursor:
            query = sql % tuple([force_text(Literal(param).as_string(cursor.cursor.cursor)) for param in params])
            cursor.execute('SELECT count_estimate(%s)', [query])
            row = cursor.fetchall()
        count = row[0][0]
        return int(count), True

    def get_count(self):
        count = None
        if self.could_use_estimate():
            approx_count, is_estimated = self.get_estimated_count()
            self.count_is_estimated = is_estimated and approx_count > settings.IDM_PAGINATOR_ESTIMATION_THRESHOLD
            if self.count_is_estimated:
                count = approx_count
        if count is None:
            count = super(EstimatingPaginator, self).get_count()
        return count

    def page(self):
        result = super(EstimatingPaginator, self).page()
        is_approx = self.count_is_estimated
        if is_approx is not None:
            result['meta']['count_is_estimated'] = is_approx
        return result


class OptimizedPaginator(EstimatingPaginator):

    def __init__(self, *args, **kwargs):
        super(OptimizedPaginator, self).__init__(*args, **kwargs)
        self.order_key_name = None
        self.last_returned_key_value = None
        self.descending_order = None

    def get_slice(self, limit, offset):
        # Возвращаем объект среза, оптимизировано если выборка отсортирована по id
        # Вызывается методом page базового класса
        if limit == 0:
            return []

        # Получим из self.objects и установим ключ сортировки в выборке
        self.set_keys()
        if self.order_key_name is None:
            # Если ключ сортировки не установлен, то возвращаем метод базового класса
            return super(OptimizedPaginator, self).get_slice(limit, offset)
        elif self.last_returned_key_value is None:
            slice_ = super(EstimatingPaginator, self).get_slice(limit, offset)
        else:
            if self.descending_order:
                slice_ = self.objects.filter(id__lt=self.last_returned_key_value)[:limit]
            else:
                slice_ = self.objects.filter(id__gt=self.last_returned_key_value)[:limit]
        if hasattr(slice_, 'unwrap'):
            slice_ = slice_.unwrap(use_random_sorting=self.use_random_sorting_in_unwrap)
            if self.nested_values:
                slice_ = slice_.nested_values(*self.nested_values)

        slice_ = list(slice_)
        if slice_:
            # Если объект среза не пустой, то устанавливаем последний возвращенный ключ
            self.last_returned_key_value = get_attribute(slice_[-1], [self.order_key_name])

        return slice_

    def generate_uri_with_last_key(self, limit, offset, last_key):
        # При генерации ссылки на следующую страницу приписываем значение ключа посследнего возвращенного объекта,
        # если оно установлено, и очищаем старое значение
        _uri = self._generate_uri(limit, offset)
        request_params = {key: value for key, value in parse_qsl(splitquery(_uri)[1])}
        if 'last_key' in request_params:
            del request_params['last_key']
        if last_key is not None:
            request_params['last_key'] = str(last_key)
        encoded_params = urlencode(request_params)
        return '%s?%s' % (self.resource_uri, encoded_params)

    def get_next(self, limit, offset, count):
        # Запрашиваем ссылку на следующую страницу
        if offset + limit >= count:
            return None
        return self.generate_uri_with_last_key(limit, offset + limit, self.last_returned_key_value)

    def get_previous(self, limit, offset):
        # Запрашиваем ссылку на предыдущую страницу
        if offset - limit < 0:
            return None
        return self.generate_uri_with_last_key(limit, offset - limit, None)

    def set_keys(self):
        # Получаем значение ключа сортировки выборки self.objects
        # Получаем последний возвращенный ключ, если он был установлен
        order_key = None
        if not isinstance(self.objects, QuerySet) or not getattr(self.objects.query, 'order_by', None):
            return
        order_by = self.objects.query.order_by[0]
        if isinstance(order_by, OrderBy):
            order_key = getattr(self.objects.query.order_by[0].expression, 'name', None)
            descending_order = getattr(self.objects.query.order_by[0], 'descending', None)
        elif isinstance(order_by, F):
            order_key = getattr(self.objects.query.order_by[0], 'name', None)
            descending_order = False
        elif isinstance(order_by, six.string_types):
            if order_by.startswith('-'):
                order_key = order_by[1:]
                descending_order = True
            else:
                order_key = order_by
                descending_order = False
        if order_key == 'id':
            order_last_key = self.request_data.get('last_key', None)
            form = IdValidationForm(objects=self.objects, key='pk', data={'query': order_last_key})
            if order_last_key is None or form.is_valid():
                self.order_key_name = order_key
                self.descending_order = descending_order
                self.last_returned_key_value = order_last_key
            else:
                raise BadRequest(_('Указанно некорректное значение ключа сортировки.'))

    def get_meta(self, limit, offset, objects=None):
        #  Не делаем лишний каунт на страницах типа запросов ролей (Их обычно меньше одной страницы)
        if objects is not None and len(objects) < limit:
            count = offset + len(objects)
        else:
            count = self.get_count()

        meta = {
            'offset': offset,
            'limit': limit,
            'total_count': count,
        }

        if limit:
            meta['previous'] = self.get_previous(limit, offset)
            meta['next'] = self.get_next(limit, offset, count)

        is_approx = self.count_is_estimated
        if is_approx is not None:
            meta['count_is_estimated'] = is_approx

        return meta

    def page(self):
        """
        Копия метода базового класса, позволяет избежать генерации meta-информации,
        если в запросе передать параметр no_meta=True
        """

        limit = self.get_limit()
        offset = self.get_offset()

        if self.request_data.get('only_meta'):
            return {'meta': self.get_meta(limit, offset)}

        objects = self.get_slice(limit, offset)
        if not isinstance(objects, list):
            objects = list(objects)

        if self.request_data.get('no_meta'):
            return {self.collection_name: objects}

        return {
            self.collection_name: objects,
            'meta': self.get_meta(limit, offset, objects),
        }
