# coding: utf-8


import logging
from typing import Iterable, List

from idm.api.exceptions import BadRequest
from idm.api.frontend.base import FrontendApiResource
from idm.api.frontend.forms import PermissionsForm


log = logging.getLogger(__name__)


class PermissionsResource(FrontendApiResource):
    """Ресурс возвращающий список пермишенов пользователя
    """

    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = None
        resource_name = 'permissions/all'
        list_allowed_methods = ['get']
        detail_allowed_methods = []

    @staticmethod
    def postprocess_permissions(permissions: Iterable[str]) -> List[str]:
        result = []
        for permission in permissions:
            if permission.startswith('idm_'):
                result.append(permission[4:])
            else:
                result.append(permission)

        return result

    def get_list(self, request, **kwargs):
        requester = self.get_requester(request)

        form = PermissionsForm(request.GET)
        if not form.is_valid():
            raise BadRequest(form.errors)

        result = requester.impersonated.get_permissions(form.cleaned_data['system'], form.cleaned_data['path'])

        return self.create_response(request, {
            'meta': {
                'total_count': len(result)
            },
            'data': self.postprocess_permissions(result),
        })
