import logging
import operator
from itertools import groupby

import ujson as json
from django.db.models import Q
from django.http import HttpResponse
from django.shortcuts import get_object_or_404
from django.utils.translation import ugettext as _
from tastypie import http, fields
from tastypie.bundle import Bundle

from idm.api.exceptions import BadRequest
from idm.api.frontend import apifields, forms
from idm.api.frontend.base import FrontendApiResource, FilterPermissionsMixin
from idm.api.frontend.forms import RoleForm, RoleDepriveForm, ReportPostForm, RoleRerequestForm
from idm.api.frontend.requesthelpers import convert_role_request_exceptions
from idm.api.frontend.serializers.light_roles_list import LightRolesListSerializer
from idm.api.frontend.serializers.roles_list import RolesListSerializer, enrich_role_fields
from idm.api.frontend.utils import OrderingAlias, fix_user_field_filters
from idm.core.constants.action import ACTION
from idm.core.constants.approverequest import APPROVEREQUEST_DECISION
from idm.core.constants.rolefield import FIELD_STATE, FIELD_TYPE
from idm.core.models import Role, RoleRequest, ApproveRequest, SystemRoleField
from idm.core.mutation import RoleFieldAppMetricaMutation, RoleFieldMetrikaCounterMutation
from idm.core.querysets.base import _can_view_all_in_system
from idm.core.utils import get_role_review_at
from idm.permissions import shortcuts as permissions_shortcuts
from idm.permissions.shortcuts import can_deprive_role
from idm.reports.tasks import make_report
from idm.users.models import User, Group, GROUP_TYPES

log = logging.getLogger(__name__)


class AbstractRoleResource(FilterPermissionsMixin, FrontendApiResource):
    """
     Абстрактный ресурс роли
    """
    group = apifields.GroupForeignKey()
    node = apifields.RoleNodeForeignKey()
    parent = apifields.ParentlessRoleForeignKey(attribute='parent', null=True, blank=True, full=True)
    system = apifields.SystemForeignKey()
    user = apifields.UserForeignKey()
    role_request = apifields.RoleRequestForRoleForeignKey(full=True, use_in='detail')

    public_roles_only = True

    role_form_class = None
    role_deprive_form_class = None
    report_post_form_class = None
    role_rerequest_form_class = None

    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = Role
        resource_name = 'roles'
        list_allowed_methods = ['get', 'post', 'delete']
        detail_allowed_methods = ['get', 'post', 'delete']
        fields = [
            'user', 'group', 'system', 'node', 'role_request',
            'fields_data', 'system_specific',
            'parent',
            'added', 'updated', 'expire_at', 'granted_at', 'review_at',
            'id', 'is_active', 'is_public', 'state',
            'state', 'ttl_date', 'ttl_days', 'review_date', 'review_days',
            'with_inheritance', 'with_robots', 'with_external', 'without_hold',
        ]
        ordering = ['state', 'added', 'updated', 'granted_at', 'expire_at', 'id', 'system', 'user', 'group']
        ordering_aliases = {
            'subject': OrderingAlias('user__username', 'group__name'),
            'system': OrderingAlias('system__name'),
            'role': OrderingAlias('node__id', 'id'),
        }
        limit = 100
        select_related_for_list = (
            'user',
            'group',
            'group__parent',
            'system',
            'node',
            'node__nodeset',
            'node__parent',
            'node__system',
            'parent',
            'parent__group',
            'parent__group__parent',
            'parent__node',
            'parent__node__system',
            'parent__node__nodeset',
            'parent__system',
            'parent__user',
            'organization'
        )
        select_related_for_detail = select_related_for_list + (
            'last_request__requester', 'user__department_group',
        )
        prefetch_related_for_detail = (
            'last_request__approves__requests__approver',
        )

    def dehydrate(self, bundle: Bundle) -> Bundle:
        bundle = super(AbstractRoleResource, self).dehydrate(bundle)

        bundle.data['human'] = bundle.obj.humanize()
        bundle.data['human_short'] = bundle.obj.humanize(format='short')
        bundle.data['human_state'] = bundle.obj.get_state_display()
        bundle.data['data'] = bundle.obj.node.data
        bundle.data['review_at'] = get_role_review_at(bundle.obj)
        return bundle

    def extract_role_request(self, bundle, requester):
        try:
            role_request = bundle.obj.get_open_request()
        except RoleRequest.DoesNotExist:
            pass
        else:
            # информацию о подтверждениях заполняем лишь для запрошенных и перезапрошенных ролей
            # сколько роли еще осталось подтверждений.
            # все индивидуальные запросы на аппрув содержат подзапросы для ИЛИ-аппруверов
            # (для которых не важно, кто именно из них подтвердил)
            bundle.data['require_approve_from'] = []

            approve_requests = (
                ApproveRequest.objects.filter(approve__role_request=role_request)
                    .select_related('approve', 'approver')
                    .order_by('approve', 'id')
            )
            for approve, requests in groupby(approve_requests, lambda ar: ar.approve):
                # собираем список ИЛИ-аппруверов для каждого незаапрувелнного аппрув-запроса.
                # если требуется аппрув от И-аппрувера - будет список из одного элемента
                require_approve_from = []
                is_approved = False
                for request in requests:
                    if request.decision != APPROVEREQUEST_DECISION.NOT_DECIDED:
                        is_approved = True
                        break
                    require_approve_from.append(apifields.ApproverApiField().convert(request.approver))
                if not is_approved:
                    bundle.data['require_approve_from'].append(require_approve_from)

            for approve_request in approve_requests:
                if approve_request.approver == requester.impersonated and approve_request.parent_id is None:
                    bundle.data['approve_request'] = {'id': approve_request.id}

    def dehydrate_for_detail(self, bundle: Bundle) -> Bundle:
        requester = self.get_requester(bundle.request)
        bundle.data['permissions'] = {
            'can_be_deprived': bool(can_deprive_role(
                requester,
                bundle.obj,
                for_api=True,
            )),
            'can_be_approved': bool(permissions_shortcuts.can_approve_role(
                requester.impersonated,
                bundle.obj,
            )),
            'can_be_rerequested': bool(permissions_shortcuts.can_rerequest_role(
                requester.impersonated,
                bundle.obj,
            )),
            'can_be_poked_if_failed': bool(permissions_shortcuts.can_retry_failed_role(
                requester.impersonated,
                bundle.obj,
            )),
        }

        bundle.data['ref_count'] = bundle.obj.refs.public().count()
        # TODO: Деприцировать и удалить этот метод
        self.extract_role_request(bundle, requester)
        return bundle

    def get_current_system(self, request):
        form = self.role_form_class(request.GET)
        if not form.is_valid():
            raise BadRequest(form.errors)

        system = form.cleaned_data.get('system')
        if system:
            return system
        return None

    def get_object_list(self, request, permission_params=None, **kwargs):
        """
        Переопределено, чтобы не создавать queryset в import-time
        """
        system = self.get_current_system(request)
        requester = self.get_requester(request)
        return (
            Role.objects
                .permitted_for(requester, system, permission_params)
                .select_related(*self._meta.select_related_for_list)
                .order_by('-updated')
        )

    def get_object_list_for_detail(self, request, permission_params=None, bypass_permissions=False, **kwargs):
        pk = kwargs['pk']
        assert pk is not None

        permission_params = permission_params or {}
        role = get_object_or_404(Role.objects.select_related('system', 'user'), pk=pk)
        permission_params['system'] = role.system
        if role.user:
            permission_params['user'] = role.user

        requester = self.get_requester(request)
        qs = Role.objects.all() if bypass_permissions else Role.objects.permitted_for(requester,
                                                                                      permission_params=permission_params)
        qs = (
            qs.
                select_related(*self._meta.select_related_for_detail).
                prefetch_related(*self._meta.prefetch_related_for_detail).
                order_by('-updated')
        )
        return qs

    def build_filters(self, request, filters=None):
        requester = self.get_requester(request)

        fix_user_field_filters(filters, ['user', 'users'])

        form = self.role_form_class(filters)
        if not form.is_valid():
            raise BadRequest(form.errors)

        query = form.cleaned_data
        qset_filters = Q()
        params = {}

        if self.public_roles_only:
            qset_filters &= Role.objects.public_query()

        if query['id']:
            params['role_ids'] = query['id']

        if query['id__gt']:
            params['role_id__gt'] = query['id__gt']

        if query['id__lt']:
            params['role_id__lt'] = query['id__lt']

        if query['sox'] is not None:
            params['sox'] = query['sox']

        if query['type']:
            params['state_set'] = query['type']

        if query['parent']:
            params['parent'] = query['parent']
        elif query['parent_type']:
            params['parent'] = query['parent_type']

        if query['state']:
            params['states'] = query['state']

        if query['ownership']:
            params['ownership'] = query['ownership']

        if 'fields_data' in query:
            params['fields_data'] = query['fields_data']

        if query['field_data']:
            if not query['system']:
                raise BadRequest(message=_('Для фильтра по полям нужно указать систему'))

            fields = query['field_data']
            # Проверяем, что для всех полей существует активный индекс
            active_fields = {
                active_field['slug']: active_field['type'] for active_field in SystemRoleField.objects.filter(
                    system=query['system'],
                    slug__in=fields.keys(),
                    state=FIELD_STATE.ACTIVE,
                ).values('slug', 'type')
            }

            missed = set(fields.keys()) - set(active_fields.keys())
            if missed:
                raise BadRequest(message=_("Для полей '%s' не найдено активных индексов" % ', '.join(missed)))

            # Значение поля уже представлено строкой,
            # поэтому переводим только нестроковые типы
            def boolean(val):
                val = val.lower()
                if val in ['true', '1']:
                    return True
                elif val in ['false', '0']:
                    return False
                raise ValueError('Invalid value %s' % val)

            fieldtypes = {
                FIELD_TYPE.BOOLEAN: boolean,
                FIELD_TYPE.INTEGER: int,
            }

            params['filter_fields_data'] = {}
            for field_name, value in fields.items():
                field_type = active_fields[field_name]
                if field_type in fieldtypes:
                    try:
                        value = fieldtypes[field_type](value)
                    except ValueError:
                        raise BadRequest(message=_(
                            """Ошибка при преобразовании значения '%(value)s' к типу '%(type)s'"""
                            % {'value': value, 'type': fieldtypes[field_type].__name__}
                        ))

                params['filter_fields_data'][field_name] = value

        if query['system']:
            params['system'] = query['system']

            if query['path']:
                params['ancestor_node'] = query['path']
                if query['role__contains']:
                    params['search_term'] = query['role__contains']

            if query['internal_role']:
                params['internal_role'] = query['internal_role']
                params['requester'] = requester

            if query['nodeset']:
                params['nodesets'] = query['nodeset']

        if filters.get('users') or filters.get('user'):
            # Роли пользователей и групп фильтруются через ИЛИ
            params['users'] = query['users'] or query['user']
        if filters.get('group'):
            # Роли пользователей и групп фильтруются через ИЛИ
            params['groups'] = query['group']

        if query['abc_slug']:
            # получим ABC сервис и все его скоупы
            groups = Group.objects.get(
                slug=f'svc_{query["abc_slug"]}',
                type=GROUP_TYPES.SERVICE,
            ).get_descendants(include_self=True)
            params['groups'] = groups

        if query['user_type']:
            params['user_type'] = query['user_type']

        params['with_parents'] = query.get('with_parents', False)

        filter_q, permission_params = Role.objects.get_filtered_query(**params)

        # На странице одного пользователя или одной группы нет смысла в count_estimate
        self.use_estimate = 'user' not in permission_params and 'group' not in permission_params
        # Всегда есть with_parents и fields_data
        self.without_filters = len(params) < 3
        qset_filters &= filter_q
        return qset_filters, permission_params

    def apply_filters(self, request, applicable_filters, permission_params=None, **kwargs):
        qs = self.get_object_list(request, permission_params=permission_params).filter(applicable_filters)
        return qs

    def post_detail(self, request, pk, **kwargs):
        role = get_object_or_404(self.get_object_list(request).select_related('system__actual_workflow'), pk=pk)
        role.lock(retry=True)

        deserialized = self.deserialize(request, request.body)

        if deserialized.get('action'):
            form_class = forms.RoleActionForm
            method = self.process_action
        else:
            # fallback на role rerequest для обратной совместимости
            form_class = self.role_rerequest_form_class
            method = self.make_transition

        form = form_class(deserialized)
        if not form.is_valid():
            raise BadRequest(form.errors)

        bundle = self.build_bundle(request=request, data=form.cleaned_data)
        method(bundle, role)

        response = self.create_response(request, None, response_class=http.HttpAccepted)
        if not hasattr(response, '_log_additional_fields'):
            response._log_additional_fields = {}
        response._log_additional_fields['system'] = role.system.slug
        return response

    def process_action(self, bundle, role):
        requester = self.get_requester(bundle.request)
        action = bundle.data.get('action')

        if action == ACTION.RETRY_FAILED:
            role.retry_failed(requester)
        else:
            raise ValueError('Action %s is not supported for role' % action)

    def make_transition(self, bundle, role):
        requester = self.get_requester(bundle.request)
        data = bundle.data

        if data.get('ttl_date') or data.get('ttl_days'):
            role.ttl_date = data['ttl_date']
            role.ttl_days = data['ttl_days']
            role.save(update_fields=['ttl_days', 'ttl_date'])

        with convert_role_request_exceptions():
            role.rerequest(
                requester=requester,
                comment=bundle.data.get('comment'),
                from_api=True,
            )

    def delete_detail(self, request, pk, **kwargs):
        """
        Отзываем роль вместо того, чтобы удалить ее
        """
        role = get_object_or_404(self.get_object_list(request), pk=pk)
        role.lock(retry=True)
        data = self.deserialize(request, request.body) if request.body else {}
        requester = self.get_requester(request)
        form = self.role_deprive_form_class(data)
        if not form.is_valid():
            raise BadRequest(form.errors)

        data = form.cleaned_data

        if role.user:
            role.user.fetch_department_group()
        # права на отъем роли проверяются внутри
        with convert_role_request_exceptions():
            role.deprive_or_decline(
                requester,
                comment=data.get('comment'),
                force_deprive=not data['validate_depriving'],
                from_api=True,
            )
        return self.create_response(request, data=None, status=204)

    def delete_list(self, request, **kwargs):
        """Отзываем роли, подпадающие под фильтр"""

        data = {}
        if hasattr(request, 'GET'):
            data = request.GET.copy()
        if request.body:
            data.update(self.deserialize(request, request.body))

        if not data or data.keys() == {'comment'}:
            raise BadRequest(_('Пожалуйста, укажите хотя бы один фильтрационный параметр'))

        requester = self.get_requester(request)
        form = self.role_deprive_form_class(data)
        if not form.is_valid():
            raise BadRequest(form.errors)

        filters, permission_params = self.build_filters(request, data)
        roles = self.apply_filters(request, filters, permission_params=permission_params).select_related(
            'user', 'user__department_group',
        )

        errors = successes = 0
        error_messages = []
        success_ids = []
        for role in roles:
            try:
                role.lock(retry=True)
                role.deprive_or_decline(
                    requester,
                    comment=form.cleaned_data.get('comment'),
                    force_deprive=not form.cleaned_data['validate_depriving'],
                    from_api=True,
                )
                successes += 1
                success_ids.append({'id': role.id})
            except Exception as e:
                log.warning('Could not deprive role %d while mass depriving', role.pk, exc_info=1)
                errors += 1
                error_messages.append({
                    'id': role.id,
                    'message': str(e),
                })
        response = {
            'successes': successes,
            'successes_ids': sorted(success_ids, key=operator.itemgetter('id')),
            'errors': errors,
            'errors_ids': sorted(error_messages, key=operator.itemgetter('id')),
        }
        return self.create_response(request, response)

    def post_list(self, request, **kwargs):
        data = self.deserialize(request, request.body)
        requester = self.get_requester(request)
        form = self.report_post_form_class(data)
        if not form.is_valid():
            raise BadRequest(form.errors)

        filters, permission_params = self.build_filters(request, data)
        roles = self.apply_filters(request, filters, permission_params).select_related('user__department_group')

        comment = form.cleaned_data.get('comment', '')
        make_report(roles, form.cleaned_data['format'], 'roles', requester.impersonated, comment=comment)

        return self.create_response(request, {
            'message': _('Отчёт формируется и будет отправлен вам на почту по завершении')
        })


class ParentlessRoleResource(AbstractRoleResource):
    parent = fields.IntegerField(attribute='parent_id', null=True, blank=True)


class RoleResource(AbstractRoleResource):
    """
     Ресурс роли
    """
    role_form_class = RoleForm
    role_deprive_form_class = RoleDepriveForm
    report_post_form_class = ReportPostForm
    role_rerequest_form_class = RoleRerequestForm
    use_random_sorting_in_unwrap = True

    def get_list(self, request, **kwargs):
        base_bundle = self.build_bundle(request=request)
        objects = self.obj_get_list(bundle=base_bundle, **self.remove_api_resource_names(kwargs))
        objects = self.apply_sorting(objects, options=request.GET)
        objects = objects.select_related(None)

        if request.GET.get('for_webauth') or request.GET.get('light'):
            serializer_class = LightRolesListSerializer
        else:
            serializer_class = RolesListSerializer

        use_random_sorting_in_unwrap = (
                self.use_random_sorting_in_unwrap and
                not (
                        'system' in self.permission_params and
                        _can_view_all_in_system(self.get_requester(request), self.permission_params['system'])
                ) and
                not self.without_filters
        )

        paginator = self._meta.paginator_class(
            request.GET,
            objects,
            resource_uri=self.get_resource_uri(),
            limit=self._meta.limit,
            max_limit=self._meta.max_limit,
            collection_name=self._meta.collection_name,
            use_estimate=self.use_estimate,
            nested_values=serializer_class.get_values_list_args(),
            use_random_sorting_in_unwrap=use_random_sorting_in_unwrap,
        )

        to_be_serialized = paginator.page()
        if self._meta.collection_name in to_be_serialized:
            serializer_class.process_list(
                to_be_serialized[self._meta.collection_name],
                resource_type=self.get_resource_type(),
            )

        return HttpResponse(content=json.dumps(to_be_serialized), content_type='application/json; charset=utf-8')


class FrontendRoleResource(RoleResource):
    @staticmethod
    def dehydrate_fields(bundle: Bundle) -> Bundle:
        if RoleFieldMetrikaCounterMutation.is_applicable(bundle.obj):
            RoleFieldMetrikaCounterMutation.mutate(bundle.data['fields_data'])
        if RoleFieldAppMetricaMutation.is_applicable(bundle.obj):
            RoleFieldAppMetricaMutation.mutate(bundle.data['fields_data'])

        enrich_role_fields(bundle.data)
        bundle.data['personal_granted_at'] = bundle.obj.get_personal_granted_at(bundle.request.GET.get('user_context'))
        return bundle

    def dehydrate_for_detail(self, bundle: Bundle) -> Bundle:
        bundle = super().dehydrate_for_detail(bundle)
        return self.dehydrate_fields(bundle)

    def dehydrate_for_related(self, bundle: Bundle) -> Bundle:
        bundle = super().dehydrate_for_related(bundle)
        return self.dehydrate_fields(bundle)
