import logging
import re
from typing import *

import waffle
from constance import config
from django.db import transaction
from django.db.models import Q
from django.http import HttpResponse, HttpRequest
from django.utils import timezone
from django.utils.translation import ugettext as _
from tastypie.bundle import Bundle
from ylog.context import log_context

from idm.api.exceptions import BadRequest, Forbidden, NotFound
from idm.api.frontend import apifields
from idm.api.frontend import forms
from idm.api.frontend.base import FrontendApiResource
from idm.api.frontend.utils import OrderingAlias
from idm.core.models import RoleNode, System
from idm.core.querysets.node import RoleNodeQueryset
from idm.core.queues import RoleNodeQueue
from idm.core.tasks.nodes import RecalcNodePipelineTask


log = logging.getLogger(__name__)
unique_id_re = re.compile(r'(?P<system_slug>[\w-]+)/unique_id/(?P<unique_id>.+)', re.U)


class RoleNodeResource(FrontendApiResource):
    """
    Ресурс узла дерева ролей
    """
    system = apifields.SystemForeignKey()

    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = RoleNode
        queryset = (
            RoleNode.objects
            .select_related(
                'parent',
                'system',
                'nodeset',
            )
            .prefetch_related(
                'fields',
                'aliases',
                'responsibilities',
                'rolenodeclosure_parents'
            )
            .order_by('id')
        )
        resource_name = 'rolenodes'
        detail_uri_name = 'slug_path'
        list_allowed_methods = ['get', 'post']
        detail_allowed_methods = ['get', 'put', 'delete']
        fields = ['id', 'state', 'slug', 'system', 'data', 'is_public', 'is_auto_updated', 'is_key',
                  'unique_id', 'slug_path', 'value_path', 'parent_path', 'review_required', 'comment_required']
        ordering = ['system']
        ordering_aliases = {
            'system': OrderingAlias('system_id', 'level', 'id'),
        }
        limit = 100

    def detail_uri_kwargs(self, bundle_or_obj):
        kwargs = {}
        obj = bundle_or_obj
        if isinstance(bundle_or_obj, Bundle):
            obj = bundle_or_obj.obj
        kwargs[self._meta.detail_uri_name] = '%s%s' % (obj.system.slug, obj.slug_path[:-1])
        return kwargs

    def _get_node_and_parent_by_unique_id(self, qs: RoleNodeQueryset, unique_id: str) -> Tuple[RoleNode, RoleNode]:
        try:
            node = qs.select_related('parent').get(unique_id=unique_id)
        except RoleNode.DoesNotExist:
            raise NotFound(message=f'Cannot find node with unique_id {unique_id}')

        return node, node.parent

    def _get_node_maybe_and_parent_by_slug_path(
            self,
            qs: RoleNodeQueryset,
            slug_path: str,
            parent_path: str) -> Tuple[Optional[RoleNode], RoleNode]:
        # Получаем родительский узел
        try:
            parent = qs.get(slug_path=parent_path)
        except RoleNode.DoesNotExist:
            raise NotFound(message='Cannot identify valid parent node from given node path')

        # Получаем (или нет) исходный узел
        try:
            old_node = qs.get(slug_path=slug_path)
        except RoleNode.DoesNotExist:
            old_node = None

        return old_node, parent

    def try_get_system_slug_and_unique_id_from_path(self, path: str) -> Tuple[Optional[str], Optional[str]]:
        match = unique_id_re.fullmatch(path)
        if not match:
            return None, None

        return match.group('system_slug'), match.group('unique_id')

    def try_get_system_slug_and_slug_path_from_path(self, path: str) -> Tuple[Optional[str], Optional[str]]:
        if '/' in path:
            slugs = path.strip('/').split('/')
            system_slug, path_slugs = slugs[0], slugs[1:]
            slug_path = '/%s/' % '/'.join(path_slugs)
        else:
            system_slug = path
            slug_path = '/'

        return system_slug, slug_path

    def get_parent_path_and_slug_from_slug_path(self, slug_path: str) -> Tuple[Optional[str], str]:
        if slug_path == '/':
            return None, ''

        slugs = list(filter(None, slug_path.split('/')))
        parents_slugs = slugs[:-1]
        slug = slugs[-1]
        if parents_slugs:
            parent_path = f'/{"/".join(parents_slugs)}/'
        else:
            parent_path = '/'

        return parent_path, slug

    def get_node_maybe(self, path: str) -> Tuple[System, RoleNode, Optional[RoleNode], str]:
        system_slug, unique_id = self.try_get_system_slug_and_unique_id_from_path(path)
        system_slug, slug_path = self.try_get_system_slug_and_slug_path_from_path(path)

        try:
            system = System.objects.get(slug=system_slug)
        except System.DoesNotExist:
            raise BadRequest(message='Cannot identify valid system from given node path')

        qs = (
            RoleNode.objects
            .get_alive_system_nodes(system)
            .select_related('system', 'nodeset')
            .prefetch_related('responsibilities__user')
        )

        if unique_id:
            old_node, parent = self._get_node_and_parent_by_unique_id(qs, unique_id)
            slug = old_node.slug if old_node else ''
        else:
            parent_path, slug = self.get_parent_path_and_slug_from_slug_path(slug_path)
            old_node, parent = self._get_node_maybe_and_parent_by_slug_path(qs, slug_path, parent_path)

        return system, parent, old_node, slug

    def get_node(self, path):
        system, parent, old_node, slug = self.get_node_maybe(path)
        if old_node is None:
            raise NotFound(message='Node could not be found')
        return old_node

    def check_permission(self, system, requester):
        if not system.is_permitted_for(requester, 'core.edit_role_nodes'):
            raise Forbidden(_('У вас нет прав на редактирование дерева ролей'))

        if system.roletree_policy == 'noneditable':
            raise Forbidden(_('Дерево этой системы не редактируется через API'))

    def get_detail(self, request, *args, **kwargs):
        """
        Получить ноду по её slug_path
        """
        node = self.get_node(kwargs['slug_path'])

        bundle = self.build_bundle(obj=node, request=request)
        bundle = self.full_dehydrate(bundle)
        bundle = self.alter_detail_data_to_serialize(request, bundle)
        return self.create_response(request, bundle)

    def build_filters(self, request, filters=None):
        form = forms.RoleNodeForm(filters)
        if not form.is_valid():
            raise BadRequest(form.errors)

        query = form.cleaned_data
        kwargs = {}

        if query['system'] is not None:
            kwargs['system'] = query['system']

        if query['slug_path'] is not None:
            kwargs['rolenodeclosure_parents__parent'] = query['slug_path']

        if query['is_key'] is not None:
            kwargs['is_key'] = query['is_key']

        if query['is_public'] is not None:
            kwargs['is_public'] = query['is_public']

        if query['updated__since'] is not None:
            kwargs['updated_at__gte'] = query['updated__since']

        if query['updated__until'] is not None:
            kwargs['updated_at__lte'] = query['updated__until']

        if query['parent'] is not None:
            kwargs['parent'] = query['parent']
        state = query.get('state')
        if not state:
            # по умолчанию отдаем только активные узлы
            kwargs['state'] = 'active'
        elif state != 'all':
            kwargs['state'] = state

        return Q(**kwargs)

    def apply_filters(self, request, applicable_filters, **kwargs):
        return self.get_object_list(request).filter(applicable_filters)

    def dehydrate(self, bundle):
        bundle.data['name'] = bundle.obj.get_name()
        bundle.data['description'] = bundle.obj.get_description()
        bundle.data['set'] = bundle.obj.nodeset.set_id if bundle.obj.nodeset_id else None
        bundle.data['human'] = bundle.obj.humanize()
        bundle.data['human_short'] = bundle.obj.humanize(format='short')
        return bundle

    def deserialize(self, request, data, *args, **kwargs):
        result = super(RoleNodeResource, self).deserialize(request, data, *args, **kwargs)
        result['node_aliases'] = result.get('aliases')
        result['node_fields'] = result.get('fields')
        result['node_responsibilities'] = result.get('responsibilities')

        # MultiValueDict возвращает get и pop по-разному
        result.pop('aliases', None)
        result.pop('fields', None)
        result.pop('responsibilities', None)
        return result

    def _init_node_modification_info(self, request, **kwargs):
        requester = self.get_requester(request)
        impersonator = requester.impersonator.username if requester.impersonator else '<UNDEFINED>'
        impersonated = requester.impersonated.username if requester.impersonated else '<UNDEFINED>'
        request.node_modification_info = {
            'method': request.method,
            'impersonator': impersonator,
            'impersonated': impersonated,
        }
        request.node_modification_info.update(kwargs)

    def post_list(self, request, **kwargs):
        """добавление новой ноды"""
        self._init_node_modification_info(request, was_restored=False, was_moved=False)
        data = self.deserialize(request, request.body)
        form = forms.CreateRoleNodeForm(data)
        if not form.is_valid():
            raise BadRequest(form.errors)
        system = form.cleaned_data.get('system')
        unique_id = data.get('unique_id')
        request.node_modification_info['has_unique_id'] = bool(unique_id)
        if waffle.switch_is_active('idm.move_node_via_post_denied') and unique_id:
            existing_node = system.nodes.active().filter(
                unique_id=unique_id).order_by('-updated_at').first()
            if existing_node is not None:
                raise BadRequest(message=f'Found active node with same unique_id: {unique_id}, system: {system.slug}')
        return self._upsert(request, system, form, create=True)

    def put_detail(self, request, **kwargs):
        """обновляем информацию про ноду path"""
        self._init_node_modification_info(request)
        system, parent, old_node, slug = self.get_node_maybe(kwargs['slug_path'])
        data = self.deserialize(request, request.body)
        form = forms.UpdateRoleNodeForm(data, system=system, instance=old_node, parent=parent, slug=slug)
        if not form.is_valid():
            raise BadRequest(form.errors)
        request.node_modification_info['can_upsert'] = form.cleaned_data.get('create', False)
        request.node_modification_info['was_renamed'] = (
            bool(data.get('slug'))
            and bool(old_node)
            and data['slug'] != old_node.slug
        )
        request.node_modification_info['was_moved'] = (
            bool(form.cleaned_data.get('parent'))
            and bool(old_node)
            and form.cleaned_data['parent'].slug_path != parent.slug_path
        )

        if form.cleaned_data.get('create', False):
            if parent.roles.returnable().exists():
                raise BadRequest(message='Parent node already has active or soon-to-be active roles')
        if 'slug' not in data and old_node is None:
            form.cleaned_data['slug'] = slug
        if not form.cleaned_data.get('parent'):
            form.cleaned_data['parent'] = parent

        request.node_modification_info['was_created'] = not bool(old_node)
        if old_node:  # Изменяем узел
            response = self._upsert(request, system, form, create=False, old_node=old_node)
        else:  # Возможно (create=true) создаём новый узел
            if not form.cleaned_data.get('create', False):
                raise BadRequest(message='Role node with given slug_path does not exist')
            response = self._upsert(request, system, form, create=True)
        return response

    def delete_detail(self, request, **kwargs):
        """удаление ноды и всего дочернего"""
        requester = self.get_requester(request)
        node = self.get_node(kwargs['slug_path'])

        self.check_permission(node.system, requester)

        depriving_nodes = node.mark_depriving(requester, immediately=True)
        for rolenode in depriving_nodes:
            rolenode.deprive(requester, from_api=True)

        log.info(
            'RoleNode with id %s deleted through API. Impersonator: %s, impersonated: %s.',
            node.id, requester.impersonator, requester.impersonated
        )
        return self.create_response(request, None, status=204)

    def _upsert(self,
                request: HttpRequest,
                system: System,
                form: forms.BaseUpdateRoleNodeForm,
                create: bool,
                old_node: Optional[RoleNode] = None) -> HttpResponse:
        requester = self.get_requester(request)
        self.check_permission(system, requester)

        form.cleaned_data['parent'].fetch_system()
        queue = RoleNodeQueue(system=system)

        if create:
            role_path = '%s%s/' % (form.cleaned_data['parent'].slug_path, form.cleaned_data['slug'])
            depriving_clone = (
                system
                .nodes
                .depriving()
                .filter(
                    depriving_at__gt=timezone.now(),  # если depriving_at в прошлом, то уже начали отзываться роли
                    slug_path=role_path,
                )
                .order_by('-updated_at')
                .first()
            )
            if depriving_clone is not None:
                log.info(
                    'Found a duplicate node while adding node %s in system %s by api',
                    depriving_clone.slug_path,
                    system.slug,
                )
                depriving_clone.mark_active()
                request.node_modification_info['was_restored'] = True
                with log_context(node_modification_info=request.node_modification_info):
                    log.info(
                        'RoleNode restored through API, id=%s',
                        depriving_clone.id
                    )
                return self.create_response(request, None, status=201)

            queue.push_addition(
                node=form.cleaned_data['parent'],
                child_data=form.as_canonical(),
                in_auto_mode=True,
            )

            queue.apply(user=requester.impersonated, from_api=True, run_node_pipeline=False)
            nodes = [item.new_node for item in queue if isinstance(item, queue.addition)]
            if request.method == 'PUT':
                assert len(nodes) == 1, 'There can be only one created node'
            added_ids = [node.id for node in nodes]
            if any((node.moved_from_id is not None) for node in nodes):
                request.node_modification_info['was_moved'] = True
            with log_context(node_modification_info=request.node_modification_info):
                log.info(
                    'RoleNode(s) added through API, id(s)=%s',
                    ', '.join(str(id) for id in added_ids)
                )
            RoleNode.objects.filter(pk__in=added_ids).update(pushed_at=timezone.now())
            response = self.create_response(request, None, status=201)
        else:
            success_status_code = 201 if form.cleaned_data.get('create', False) else 204
            if not old_node:
                raise BadRequest(message='Node was not found')

            queue.push_modification(
                node=old_node,
                new_data=form.as_canonical(),
                in_auto_mode=True,
            )
            queue.apply(user=requester.impersonated, from_api=True, run_node_pipeline=False)

            if form.cleaned_data['parent'] and form.cleaned_data['parent'] != old_node.parent:
                old_node.move(form.cleaned_data['parent'], requester)
                old_node.get_descendants(include_self=True).update(moved_at=timezone.now())

            with log_context(node_modification_info=request.node_modification_info):
                log.info('RoleNode modified through API, id=%s', old_node.id)
            response = self.create_response(request, None, status=success_status_code)

        if system.slug not in config.RECALC_PIPELINE_IN_API_BLACKLIST.split(','):
            transaction.on_commit(lambda: RecalcNodePipelineTask.delay(system_id=system.pk))
        return response
