import logging

from django.db.models import Q
from django.utils.translation import ugettext as _
from tastypie import fields

from idm.api.exceptions import BadRequest, Forbidden
from idm.api.frontend import apifields
from idm.api.frontend.base import FrontendApiResource
from idm.api.frontend.forms import RoleRequestFilterForm, RoleRequestForm
from idm.api.frontend.requesthelpers import convert_role_request_exceptions
from idm.api.frontend.role import RoleResource
from idm.core.models import RoleRequest, Role

log = logging.getLogger(__name__)


class RoleRequestResource(FrontendApiResource):
    """
    Ресурс запросов на получение ролей пользователями
    """
    requester = apifields.UserForeignKey(attribute='requester', replace_none_with_robot=True)
    approves = fields.ToManyField('idm.api.frontend.approve.ApproveResource', 'approves', full=True)
    role = apifields.RoleForeignKey()

    public_roles_only = True

    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = RoleRequest
        resource_name = 'rolerequests'
        list_allowed_methods = ['get', 'post']
        detail_allowed_methods = ['get']
        fields = ['id', 'added', 'is_done', 'notify_everyone', 'approves', 'requester', 'role',
                  'with_inheritance', 'with_robots', 'with_external', 'without_hold']

    # класс ресурса роли нужен, чтобы после запроса роли отдать её detail данные
    role_resource_class = RoleResource

    def get_object_list(self, request, **kwargs):
        """
        Переопределено, чтобы не создавать queryset в import-time.
        """
        requester = self.get_requester(request)

        # todo: учесть impersonator
        role_requests = (
            RoleRequest.objects.filter(requester=requester.impersonated).
            select_related(
                'role__user',
                'role__group',
                'role__system',
                'role__organization',
                'role__node__system',
                'role__parent__user',
                'role__parent__group',
                'role__parent__system',
                'role__parent__node__system',
                'requester',
                'workflow',
                'workflow__approver',
                'workflow__user',
                'workflow__system',
            ).
            prefetch_related(
                'approves__requests__approver'
            ).
            order_by('-pk')
        )

        return role_requests

    def build_filters(self, request, filters=None):
        form = RoleRequestFilterForm(filters)
        if not form.is_valid():
            raise BadRequest(form.errors)

        query = form.cleaned_data
        qset_filters = Q()

        if self.public_roles_only:
            qset_filters &= Role.objects.public_query(prefix='role__')

        if query['requester']:
            qset_filters &= Q(requester=query['requester'])

        if query['system']:
            qset_filters &= Q(role__system=query['system'])

        if query['path']:
            qset_filters &= Q(role__node__rolenodeclosure_parents__parent=query['path'])

        if query['is_done'] is not None:
            qset_filters &= Q(is_done=query['is_done'])

        return qset_filters

    def apply_filters(self, request, applicable_filters, **kwargs):
        return self.get_object_list(request).filter(applicable_filters)

    def post_list(self, request, **kwargs):
        """
        Создать (либо проверить возможность создания) запрос на выдачу роли
        """
        # запрашивающий роль берется из кук
        post_data = self.deserialize(request, request.body)
        requester = self.get_requester(request, post_data)

        form = RoleRequestForm(post_data)
        if not form.is_valid():
            raise BadRequest(form.errors)

        subject = form.get_subject()
        data = form.cleaned_data

        # Запрашивать скрытые роли из frontend нельзя
        # Из v1 - можно только при наличии соответствующего гранта
        # RULES-2895
        if self.public_roles_only:
            can_request_role = data['path'].is_public
        else:
            can_request_role = (data['path'].is_public
                                or data['system'].is_permitted_for(requester, 'core.request_role'))
        if not can_request_role:
            # ugettext_lazy использовать нельзя, потому что его не понимает RestApiError
            raise Forbidden(_('Вы не можете запрашивать скрытую роль'))

        if data['path'].comment_required and not data.get('comment'):
            raise BadRequest(message=_('Эта роль требует обязательного комментария'))

        simulate = data['simulate']

        with convert_role_request_exceptions(simulated=simulate):
            if simulate:
                approvers_groups, additional = Role.objects.simulate_role_request(
                    requester=requester,
                    subject=subject,
                    **extract_kwargs_for_role_request(data),
                )
                simulated = {
                    'simulated': True,
                    'approvers': [
                        # в group лежат объекты Approver
                        [apifields.ApproverApiField().convert(approver.user) for approver in group]
                        for group in approvers_groups
                    ]
                }
                simulated.update(additional)
                response = self.create_response(request, simulated)
            else:
                role = Role.objects.request_role(
                    requester=requester,
                    subject=subject,
                    from_api=True,
                    **extract_kwargs_for_role_request(data),
                )
                if data['no_meta']:
                    response = self.create_response(request, {'id': role.id})
                else:
                    response = self.role_resource_class().get_detail(request, pk=role.id, bypass_permissions=True)
                response.status_code = 201
        if not hasattr(response, '_log_additional_fields'):
            response._log_additional_fields = {}
        response._log_additional_fields['system'] = data['system'].slug

        return response


def extract_kwargs_for_role_request(data):
    args = ['system', 'comment', 'silent', 'request_fields']
    kwargs = {arg: data.get(arg) for arg in args}

    kwargs['data'] = data['path']
    kwargs['fields_data'] = data['fields_data']
    kwargs['ttl_date'] = data['deprive_at']
    kwargs['ttl_days'] = data['deprive_after_days']
    kwargs['with_inheritance'] = data.get('with_inheritance', True)
    kwargs['with_robots'] = data.get('with_robots', True)
    kwargs['with_external'] = data.get('with_external', True)
    kwargs['without_hold'] = data.get('without_hold', False)
    kwargs['review_date'] = data['review_at']

    organization = kwargs.get('organization')
    kwargs['organization_id'] = organization.id if organization else None

    return kwargs


class RoleRequestShortResource(RoleRequestResource):
    """Ресурс запроса роли без запросившего, роли и workflow для ручки approverequests"""
    class Meta(RoleRequestResource.Meta):
        fields = ['id', 'added', 'is_done', 'approves']


class RoleRequestForRoleResource(RoleRequestResource):
    """Ресурс запроса роли без роли для ручки ролей"""
    class Meta(RoleRequestResource.Meta):
        fields = ['id', 'added', 'is_done', 'approves', 'requester']
