# coding: utf-8

import logging

from idm.api.exceptions import BadRequest
from idm.api.frontend.base import FrontendApiResource
from idm.api.frontend.forms import RoleAdditionalFieldsRequestForm
from idm.core.constants.system import SYSTEM_CHECKS, SYSTEM_GROUP_POLICY
from idm.users.constants.group import GROUP_TYPES

log = logging.getLogger(__name__)


class RoleRequestAdditionalFieldsResource(FrontendApiResource):
    """
    Ресурс, принимающий систему и отдающий список полей-галок
    """

    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = None
        resource_name = 'rolerequestadditionalfields'
        list_allowed_methods = ['get']
        detail_allowed_methods = []
        include_resource_uri = False

    def obj_get_list(self, bundle, **kwargs):
        request = bundle.request.GET
        form = RoleAdditionalFieldsRequestForm(request)
        if not form.is_valid():
            raise BadRequest(form.errors)

        data = form.cleaned_data
        return self.fields_for_slugs(self.system_checks_slugs(data))

    @staticmethod
    def fields_for_slugs(slugs):
        fields = [
            {
                'slug': slug,
                'name': SYSTEM_CHECKS.NAMES[slug],
                'type': 'bool',
            }
            for slug in slugs
        ]
        return fields

    @staticmethod
    def system_checks_slugs(data):
        group = data['group']
        system = data['system']
        if (system.group_policy != SYSTEM_GROUP_POLICY.UNAWARE or
                group is None or
                group.type not in GROUP_TYPES.USER_GROUPS):
            result = []
        else:
            result = [SYSTEM_CHECKS.WITH_ROBOTS, SYSTEM_CHECKS.WITH_EXTERNAL, SYSTEM_CHECKS.WITHOUT_HOLD]
            if group.type == GROUP_TYPES.DEPARTMENT:
                result.append(SYSTEM_CHECKS.WITH_INHERITANCE)
        return result

    def dehydrate(self, bundle):
        bundle.data = bundle.obj
        return bundle
