# coding: utf-8

import logging
from django.http import Http404

from idm.api.exceptions import BadRequest
from idm.api.frontend.base import FrontendApiResource
from idm.api.frontend.forms import RoleFieldsRequestForm
from idm.core.models import RoleField

log = logging.getLogger(__name__)


class RoleRequestFieldsResource(FrontendApiResource):
    """
    Ресурс, принимающий узел дерева ролей и отдающий список полей
    """

    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = RoleField
        resource_name = 'rolerequestfields'
        list_allowed_methods = ['post']
        detail_allowed_methods = []
        include_resource_uri = False

    def post_list(self, request, **kwargs):
        """Имитируем GET-запрос через POST, чтобы удобно передавать данные формы"""
        return self.get_list(request, **kwargs)

    def obj_get_list(self, bundle, **kwargs):
        deserialized = self.deserialize(bundle.request, bundle.request.body)
        form = RoleFieldsRequestForm(deserialized)
        if not form.is_valid():
            raise BadRequest(form.errors)

        data = form.cleaned_data

        role_node = data['path']
        fields_data = data['fields_data']
        if fields_data is None:
            fields_data = {}

        if not role_node.is_requestable():
            raise Http404('Role node is not found')

        fields = role_node.get_fields(fields_data)
        ignored_types = RoleField.IGNORED_USER_FIELD_TYPES if data.get('user') else RoleField.IGNORED_GROUP_FIELD_TYPES
        fields = [field for field in fields if field.type not in ignored_types]

        return fields

    def dehydrate(self, bundle):
        bundle.data = bundle.obj.as_frontend_api()
        return bundle
