from typing import Dict, Any, Optional, Collection

from idm.utils.i18n import get_lang_key


class Serializer:
    db_fields = []
    datetime_fields = []
    complex_fields = {}
    fields_to_delete = []

    @classmethod
    def process(cls, obj: Dict[str, Any], **kwargs):
        if not isinstance(obj, dict):
            return

        cls.process_obj_fields(obj, **kwargs)
        cls.process_complex_fields(obj, **kwargs)
        cls.delete_extra_fields(obj, **kwargs)
        cls.process_datetime_fields(obj, **kwargs)
        cls.postprocess(obj, **kwargs)

    @classmethod
    def postprocess(cls, obj, **kwargs):
        pass

    @classmethod
    def process_datetime_fields(cls, obj, **kwargs):
        for field in cls.datetime_fields:
            if obj[field] is not None:
                obj[field] = obj[field].isoformat()

    @classmethod
    def delete_extra_fields(cls, obj, **kwargs):
        for field in cls.fields_to_delete:
            obj.pop(field, None)

    @classmethod
    def process_complex_fields(cls, obj, **kwargs):
        for field, serializer in cls.complex_fields.items():
            serializer.process(obj[field], **kwargs)

    @classmethod
    def process_obj_fields(cls, obj, **kwargs):
        pass

    @classmethod
    def get_values_list_args(cls):
        fields = cls.db_fields[:]

        for field, serializer in cls.complex_fields.items():
            for nested_field in serializer.get_values_list_args():
                fields.append(f'{field}__{nested_field}')

            fields.append(field)

        return fields


class ListSerializer:
    obj_serializer = None

    @classmethod
    def process_list(
            cls,
            objs: Collection[Dict[str, Any]],
            lang: Optional[str] = None,
            resource_type: Optional[str] = None,
            **kwargs,
    ):
        lang = lang or get_lang_key()
        for obj in objs:
            cls.obj_serializer.process(obj, lang=lang, resource_type=resource_type, **kwargs)

    @classmethod
    def get_values_list_args(cls):
        return cls.obj_serializer.get_values_list_args()
