# coding: utf-8

import logging
import re

import waffle
from django import forms as django_forms
from django.db.models import Q
from django.utils.encoding import force_text
from intrasearch_fetcher.exceptions import IntrasearchFetcherException

from idm.api.exceptions import BadRequest, InternalAPIError
from idm.api.frontend import forms
from idm.api.frontend.base import FrontendApiResource
from idm.sync.intrasearch import IntrasearchFetcher
from idm.users.constants.user import USER_TYPES
from idm.users.models import Group, Subject, User
from idm.utils.i18n import get_lang_key


logger = logging.getLogger(__name__)
login_re = re.compile(r'^[a-z0-9_\.\-]*$')


class SuggestNotFound(Exception):
    pass


class BaseSuggestResource(FrontendApiResource):
    id_field = django_forms.IntegerField()
    id_field_name = 'pk'
    override_query = None
    raise_not_found = False

    class Meta(object):
        object_class = None
        list_allowed_methods = ['get']
        detail_allowed_methods = []
        form = forms.SuggestForm

    def get_list(self, request, **kwargs):
        if not self.override_query:
            form = self._meta.form(request.GET)

            if not form.is_valid():
                raise BadRequest(form.errors)

            self.query = form.cleaned_data
        else:
            self.query = self.override_query
        result = self.objects_pipeline(request)

        return self.create_response(
            request,
            {
                'meta': {'limit': self.query['limit'], 'offset': self.query['offset']},
                'data': result,
            },
        )

    def objects_pipeline(self, request):
        objects = self.get_object_list(request)
        objects = self.filter_objects(objects, request)
        objects = self.limit_objects(objects, request)
        if self.is_objects_empty(objects) and self.raise_not_found:
            raise SuggestNotFound()
        return self.process_objects(objects, request)

    @staticmethod
    def is_objects_empty(objects):
        return not objects

    def filter_by_id(self, objects):
        pks = []
        if self.query.get('id'):
            pks.append(self.query['id'])
        elif self.query.get('id__in'):
            pks.extend(self.query['id__in'].split(','))

        try:
            pks = list(map(self.id_field.to_python, pks))
        except django_forms.ValidationError as e:
            raise BadRequest(message={self.id_field_name: force_text(e.message)})

        q = Q(**{'{}__in'.format(self.id_field_name): pks}) if pks else Q()
        return objects.filter(q)

    def filter_objects(self, objects, request):
        return self.filter_by_id(objects)

    def process_objects(self, objects, request):
        return objects

    def limit_objects(self, objects, request):
        return objects[self.query['offset']:self.query['offset'] + self.query['limit']]


class BaseSubjectSuggestResource(BaseSuggestResource):
    fullname_fields = ['full_name']
    fullname_en_fields = ['full_name_en']
    ident_field = 'username'
    id_field = django_forms.CharField()
    id_field_name = 'external_id'

    def filter_users(self, objects, query):
        if not query:
            return objects

        username_params = []
        fullname_params = []
        fullname_en_params = []
        parts = [part.strip().lower() for part in query.split()]
        parts = [_f for _f in parts if _f]
        for part in parts:
            if login_re.match(part):
                username_params.append(part)
            else:
                fullname_params.append(part)
            fullname_en_params.append(part)
        ident_q = Q()
        for term in username_params:
            ident_q |= Q(**{'%s__icontains' % self.ident_field: term})
        name_q = Q()
        for term in fullname_params:
            q = Q()
            for fieldname in self.fullname_fields:
                q |= Q(**{'%s__icontains' % fieldname: term})
            name_q &= q
        name_en_q = Q()
        for term in fullname_en_params:
            q = Q()
            for fieldname in self.fullname_en_fields:
                q |= Q(**{'%s__icontains' % fieldname: term})
            name_en_q &= q
        objects = objects.filter(ident_q | name_q | name_en_q)

        return objects

    def filter_tvm_by_service(self, objects, query):
        if not query:
            return objects

        groups = Group.objects.tvm_groups()
        groups = groups.filter(
            Q(slug__icontains=query) |
            Q(name__icontains=query) |
            Q(name_en__icontains=query)
        )
        memberships_q = Q(memberships__group__in=groups, memberships__state='active')
        user_model = objects.model
        if issubclass(user_model, Subject):
            objects |= user_model.objects.filter(
                user__in=User.objects.filter(memberships_q, type=USER_TYPES.TVM_APP)
            )
        else:
            objects |= user_model.objects.filter(memberships_q)

        return objects

    def filter_by_type(self, objects, type):
        raise NotImplementedError

    def filter_objects(self, objects, request):
        objects = super(BaseSubjectSuggestResource, self).filter_objects(objects, request)
        objects = self.filter_users(objects, self.query['q'])
        type = self.query['type'] or USER_TYPES.USER
        if type == USER_TYPES.TVM_APP:
            objects = self.filter_tvm_by_service(objects, self.query['q'])

        objects = self.filter_by_type(objects, type)

        return objects


class ProxiedSuggestResource(BaseSuggestResource):

    def __init__(self, *args, **kwargs):
        self.fetcher = IntrasearchFetcher(self._meta.layer)

    def limit_objects(self, objects, request):
        return objects

    def filter_objects(self, objects, request):
        return objects

    def process_objects(self, objects, request):
        return [self.process_object(obj) for obj in objects]

    def process_object(self, obj):
        return obj

    def get_filters(self):
        return {}

    def get_object_list(self, request, **kwargs):
        try:
            return self.fetcher.search(
                text=self.query['q'],
                filters=self.get_filters(),
                language=get_lang_key(),
                limit=self.query['limit'],
                offset=self.query['offset'],
            )
        except IntrasearchFetcherException as err:
            raise InternalAPIError(*err.args)


class SwitchResource(FrontendApiResource):

    class Meta(FrontendApiResource.Meta):
        abstract = False
        object_class = None
        switch_options = {}
        switch_name = None

    def use_proxied(self, request_args: dict):
        use_proxied = waffle.switch_is_active(self._meta.switch_name)
        has_search = request_args.get('q', None)
        has_direct = request_args.get('id', None)
        if has_direct or not has_search:
            # Если поиск по идентификатору или без строки поиска, то используем наш саджест,
            # так как он быстрее в этих случаях, и при этом нет риска потерять ранжирование,
            # так как в случае id в выборке всегда один элемент, а если нет поиска, то и ранжирования нет
            use_proxied = False

        return use_proxied

    def get_resource(self, request_args: dict):
        return self._meta.switch_options[self.use_proxied(request_args)]()

    def dispatch_list(self, request, *args, **kwargs):
        resource = self.get_resource(request.GET)
        if request.GET.get('id__in'):
            resource.raise_not_found = True
            try:
                return resource.dispatch_list(request, *args, **kwargs)
            except SuggestNotFound:
                query = resource.query
                query['q'] = query['id__in']
                query['id__in'] = ''
                resource = self.get_resource(query)
                resource.override_query = query

        return resource.dispatch_list(request, *args, **kwargs)

    def dispatch_detail(self, request, *args, **kwargs):
        resource = self.get_resource(request)
        return resource.dispatch_detail(request, *args, **kwargs)


class SuggestSwitchResource(SwitchResource):

    class Meta(SwitchResource.Meta):
        list_allowed_methods = ['get']
        detail_allowed_methods = []
        switch_name = 'idm.proxied_suggest'


class SuggestTVMSwitchResource(SuggestSwitchResource):
    class Meta(SuggestSwitchResource.Meta):
        pass

    def use_proxied(self, request_args: dict):
        use_proxied = super(SuggestTVMSwitchResource, self).use_proxied(request_args)
        if use_proxied:
            use_proxied = request_args.get('type') != USER_TYPES.TVM_APP

        return use_proxied
