# coding: utf-8


import logging

from django import forms

from idm.api.frontend.suggest.base import (
    SuggestSwitchResource,
    BaseSubjectSuggestResource,
    ProxiedSuggestResource
)
from idm.users.models import Group

logger = logging.getLogger(__name__)


class ProxiedGroupSuggestResource(ProxiedSuggestResource):

    class Meta(ProxiedSuggestResource.Meta):
        layer = 'idm_groups'

    def process_object(self, obj):
        return {
            'full_name': obj['title'],
            'group_type': obj['fields'].get('type'),
            'id': int(obj['id']),
            'is_group': True
        }

    def process_objects(self, objects, request):
        objects = super(ProxiedGroupSuggestResource, self).process_objects(objects, request)
        ids = [obj['id'] for obj in objects]

        groups = (
            Group.objects
            .filter(external_id__in=ids)
            .select_related('parent')
        )

        parent_names = {
            group.external_id: group.parent.get_name() if group.parent else ''
            for group in groups
        }

        for obj in objects:
            obj['department__name'] = parent_names[obj['id']]

        return objects


class LocalGroupSuggestResource(BaseSubjectSuggestResource):
    fullname_fields = ['name']
    fullname_en_fields = ['name_en']
    ident_field = 'external_id'
    id_field = forms.IntegerField()
    id_field_name = 'external_id'

    def filter_by_type(self, objects, type):
        # Пока что не поддерживаем фильтр по типу
        return objects

    def process_objects(self, objects, request):
        return [self.process_object(obj, request) for obj in objects]

    def process_object(self, obj, request):
        return {
            'id': obj.external_id,  # username для пользователей, external_id для групп
            'is_group': True,
            'group_type': obj.type,
            'department__name': obj.parent.get_name() if obj.parent else '',
            'full_name': obj.get_name(),
        }

    def get_object_list(self, request, **kwargs):
        return Group.objects.user_groups().filter(parent__isnull=False).active().order_by('name').select_related('parent')


class GroupSuggestSwitchResource(SuggestSwitchResource):

    class Meta(SuggestSwitchResource.Meta):
        resource_name = 'suggest/groups/all'
        switch_options = {
            True: ProxiedGroupSuggestResource,
            False: LocalGroupSuggestResource
        }
