# coding: utf-8


import logging

from django import forms as django_forms
from django.conf import settings
from django.db.models import Q
from guardian.shortcuts import get_objects_for_user

from idm.api.frontend import forms
from idm.api.frontend.suggest.base import BaseSuggestResource
from idm.core.models import InternalRole
from idm.utils.i18n import get_lang_key

logger = logging.getLogger(__name__)


class BaseInternalRoleSuggestResource(BaseSuggestResource):
    mapping = dict(settings.IDM_SYSTEM_ROLES)
    id_field = django_forms.CharField()
    id_field_name = 'role'

    def filter_objects(self, objects, request):
        objects = super(BaseInternalRoleSuggestResource, self).filter_objects(objects, request)
        if self.query['q']:
            objects = objects.filter(Q(role__icontains=self.query['q']) |
                                     Q(role__in=[name for name, role in self.mapping.items()
                                                 if (self.query['q'].lower() in role['name']['ru'].lower() or
                                                     self.query['q'].lower() in role['name']['en'].lower())]))
        return objects.values_list('role', flat=True).distinct()

    def process_objects(self, objects, request):
        result = [{
            'id': role_name,
            'name': self.mapping[role_name]['name'][get_lang_key()]
        } for role_name in objects]
        return result


class AllInternalRoleSuggestResource(BaseInternalRoleSuggestResource):
    class Meta:
        object_class = None
        form = forms.SuggestByRequiredSystemForm
        resource_name = 'suggest/internalroles/all'

    def get_object_list(self, request, **kwargs):
        requester = self.get_requester(request)

        return (get_objects_for_user(requester.impersonated, 'core.idm_view_roles', klass=InternalRole,
                                     use_groups=False, with_superuser=False).
                filter(node__system=self.query['system'], node__level__gt=1).order_by('role'))
