import logging
from collections import Counter
from typing import List

from django import forms as django_forms
from django.conf import settings
from django.core.cache import cache
from django.http import HttpRequest

from idm.api.exceptions import BadRequest
from idm.api.frontend import forms
from idm.api.frontend.suggest.base import SuggestSwitchResource, BaseSuggestResource, ProxiedSuggestResource
from idm.core.models import RoleNode
from idm.core.querysets.node import RoleNodeQueryset

logger = logging.getLogger(__name__)


def get_response(node, objects=None):
    key_node = None
    try:
        key_node = node.get_child_nodes().get()
    except node.DoesNotExist:
        header = None
    except node.MultipleObjectsReturned:
        raise BadRequest('Cannot suggest key roles')
    else:
        header = {
            'name': key_node.get_name(),
            'path': node.value_path,
            'slug': key_node.slug
        }
    if objects is None:
        if key_node is not None:
            objects = key_node.get_child_nodes().public()
            # для ускорения выборок
            objects = objects.filter(level=key_node.level+1, system_id=key_node.system_id)
        else:
            objects = RoleNode.objects.none()

    return {
        'header': header,
        'roles': objects
    }


class RoleNodeSuggestMixin(object):
    def objects_pipeline(self, request):
        objects = super(RoleNodeSuggestMixin, self).objects_pipeline(request)
        roles = objects['roles']
        names = Counter([role['name'] for role in roles])
        ids = Counter([role['id'] for role in roles])
        system_slug = self.query['system'].slug
        path = self.query['scope'].slug_path
        for name, count in names.items():
            if count > 1:
                logger.warning(
                    'Duplicate node names found in suggest. Resource: "%s", system: "%s", path: "%s", name: "%s"',
                    self.__class__.__name__,
                    system_slug,
                    path,
                    name
                )

        for idx, count in ids.items():
            if count > 1:
                logger.warning(
                    'Duplicate node ids found in suggest. Resource: "%s", system: "%s", path: "%s", id: "%s"',
                    self.__class__.__name__,
                    system_slug,
                    path,
                    idx
                )
        return objects


class ProxiedRoleNodeSuggestResource(RoleNodeSuggestMixin, ProxiedSuggestResource):

    class Meta(ProxiedSuggestResource.Meta):
        layer = 'idm_rolenodes'
        form = forms.RolesSuggestForm

    def get_filters(self):
        node = self.query['scope']
        id_ = self.query['id']

        filters = {'s_parent_path': node.self_path}

        if id_:
            filters['s_slug'] = id_

        return filters

    def process_objects(self, objects, request):
        objects = super(ProxiedRoleNodeSuggestResource, self).process_objects(objects, request)
        return get_response(self.query['scope'], objects)

    def process_object(self, obj):
        result = {
            'name': obj['title'],
            'id': obj['fields']['slug'],
            'slug': obj['fields']['slug'],
        }

        if 'help' in obj['fields']:
            result['help'] = obj['fields']['help']

        return result


class LocalRoleNodeSuggestResource(RoleNodeSuggestMixin, BaseSuggestResource):
    id_field = django_forms.CharField()
    id_field_name = 'slug'

    class Meta(BaseSuggestResource.Meta):
        form = forms.RolesSuggestForm

    @staticmethod
    def is_objects_empty(objects):
        return not objects['roles']

    def filter_objects(self, objects, request: HttpRequest):
        nodes = objects['roles'].order_by_with_l10n('name').public()
        nodes: RoleNodeQueryset = super(LocalRoleNodeSuggestResource, self).filter_objects(nodes, request)
        if query := self.query.get('q', '').lower():
            nodes = nodes.text_search(query)
        objects['roles'] = nodes
        return objects

    def limit_objects(self, objects, request: HttpRequest):
        objects['roles'] = objects['roles'][self.query['offset']:self.query['offset'] + self.query['limit']]
        return objects

    @staticmethod
    def get_suggest_items_from_queryset(nodes: RoleNodeQueryset) -> List[dict]:
        roles = []
        for node in nodes:
            role = {
                'id': node.slug,
                'slug': node.slug,
                'name': node.get_name(),
            }
            description = node.get_description()
            if description:
                role['help'] = description
            roles.append(role)

        return roles

    def maybe_cached(self, objects: RoleNodeQueryset) -> List[dict]:
        if (
            self.query['offset'] == settings.IDM_SUGGEST_OFFSET and
            self.query['limit'] == settings.IDM_SUGGEST_LIMIT and
            self.query['scope'].is_root_node() and
            not self.query['id'] and
            not self.query['id__in'] and
            not self.query['q']
        ):
            cache_key = self.query['scope'].system.suggest_cache_key
            items = cache.get(cache_key)
            if items is None:
                items = self.get_suggest_items_from_queryset(objects)
                cache.set(cache_key, items, timeout=settings.IDM_SUGGEST_CACHE_TIMEOUT)

        else:
            items = self.get_suggest_items_from_queryset(objects)

        return items

    def process_objects(self, objects, request):
        objects['roles'] = self.maybe_cached(objects['roles'])
        return objects

    def get_object_list(self, request, **kwargs):
        node = self.query['scope']
        return get_response(node)


class RoleNodeSuggestSwitchResource(SuggestSwitchResource):

    class Meta(SuggestSwitchResource.Meta):
        resource_name = 'suggest/roles/all'
        switch_options = {
            True: ProxiedRoleNodeSuggestResource,
            False: LocalRoleNodeSuggestResource
        }
