# coding: utf-8


import logging

from django.db.models import Q
from django.db.models.functions import Length

from idm.api.frontend import forms
from idm.api.frontend.suggest.base import BaseSuggestResource
from idm.services.models import Service
from idm.utils.i18n import get_localized_fieldname

logger = logging.getLogger(__name__)


class ServiceSuggestResource(BaseSuggestResource):
    id_field_name = 'external_id'

    class Meta(BaseSuggestResource.Meta):
        resource_name = 'suggest/services/all'
        form = forms.SuggestForm

    def get_object_list(self, request, **kwargs):
        return Service.objects.nonroot().order_by(get_localized_fieldname('name'))

    def filter_objects(self, objects, request):
        objects = super(ServiceSuggestResource, self).filter_objects(objects, request)
        if self.query['q']:
            objects = objects.filter(
                Q(slug__icontains=self.query['q']) |
                Q(name__icontains=self.query['q']) |
                Q(name_en__icontains=self.query['q'])
            )
        return objects.annotate(name_len=Length('name')).order_by('name_len')

    def process_objects(self, objects, request):
        return [
            {
                'id': service.external_id,
                'slug': service.slug,
                'name': service.get_name(),
            }
            for service in objects
        ]
