# coding: utf-8


import logging

from django.db.models import Q

from idm.api.frontend import fields, forms
from idm.api.frontend.suggest.base import BaseSuggestResource
from idm.core.models import System

logger = logging.getLogger(__name__)


class SystemsSuggestResource(BaseSuggestResource):
    id_field = fields.LowerCharField()
    id_field_name = 'slug'

    def filter_objects(self, objects, request):
        objects = super(SystemsSuggestResource, self).filter_objects(objects, request)
        if self.query['q']:
            objects = objects.filter(
                Q(slug__icontains=self.query['q']) |
                Q(name__icontains=self.query['q']) |
                Q(name_en__icontains=self.query['q'])
            )
        return objects

    def process_objects(self, objects, request):
        return [
            {
                'id': system.slug,
                'slug': system.slug,
                'name': system.get_name(),
                'description': system.get_description_text(),
            }
            for system in objects
        ]


class AllSystemsSuggestResource(SystemsSuggestResource):
    """
        для формы на странице пользователя (все системы в базе, если пользователь смотрит свою страницу)
    """
    class Meta(SystemsSuggestResource.Meta):
        resource_name = 'suggest/systems/all'
        form = forms.SuggestForm

    def get_object_list(self, request, **kwargs):
        return System.objects.filter(is_active=True).order_by('name')
