# coding: utf-8


import logging

from django import forms

from idm.api.frontend.suggest.base import (
    SuggestTVMSwitchResource,
    BaseSubjectSuggestResource,
    ProxiedSuggestResource
)
from idm.users.models import User

logger = logging.getLogger(__name__)


class ProxiedUserSuggestResource(ProxiedSuggestResource):

    class Meta(ProxiedSuggestResource.Meta):
        layer = 'idm_users'

    def process_object(self, obj):
        return {
            'department__name': obj['fields'].get('department_name'),
            'first_name': obj['fields'].get('first_name'),
            'full_name': obj['title'],
            'id': obj['id'],
            'is_group': False,
            'last_name': obj['fields'].get('last_name'),
            'login': obj['id']
        }


class LocalUserSuggestResource(BaseSubjectSuggestResource):
    fullname_fields = ['first_name', 'last_name']
    fullname_en_fields = ['first_name_en', 'last_name_en']
    ident_field = 'username'
    id_field_name = 'username'
    id_field = forms.CharField()

    def filter_by_type(self, objects, type):
        if not type:
            return objects

        return objects.filter(type=type)

    def process_objects(self, objects, request):
        return [self.process_object(obj, request) for obj in objects]

    def process_object(self, obj, request):
        dep_name = ''
        if obj.department_group is not None:
            dep_name = obj.department_group.get_name()
        return {
            'id': obj.username,
            'is_group': False,
            'department__name': dep_name,
            'first_name': obj.first_name,
            'full_name': obj.get_full_name(),
            'last_name': obj.last_name,
            'login': obj.username,
            'work_email': obj.email,
        }

    def get_object_list(self, request, **kwargs):
        return (
            User.objects
            .users()
            .filter(department_group__isnull=False)
            .order_by('first_name', 'last_name', 'username')
            .select_related('department_group')
        )


class UserSuggestSwitchResource(SuggestTVMSwitchResource):

    class Meta(SuggestTVMSwitchResource.Meta):
        resource_name = 'suggest/users/all'
        switch_options = {
            True: ProxiedUserSuggestResource,
            False: LocalUserSuggestResource
        }
