"""
Здесь собраны ручки для выполнения различных операций
над пачками объектов в асинхронном режиме через celery
"""
import uuid
import constance

from django.db import transaction
from django.utils.translation import ugettext as _

from idm.api.frontend.base import FrontendApiResource
from idm.api.exceptions import BadRequest, NotFound
from idm.api.v1.forms import DelayedRoleRequestForm, DelayedRoleDepriveForm
from idm.api.v1.base import RequesterMixin
from idm.core.constants.batchrequest import BATCH_REQUEST_TYPE
from idm.core.models import BatchRequest, DelayedRoleRequest
from idm.core.tasks.roles import ExecuteDelayedRoleRequests


class BaseBatchRequestResource(RequesterMixin, FrontendApiResource):

    batch_request_type = None

    class Meta(FrontendApiResource.Meta):
        object_class = BatchRequest
        list_allowed_methods = ['post']
        detail_allowed_methods = ['get']

    def get_detail(self, request, **kwargs):
        batch_id = kwargs.get('pk')

        try:
            batch_id = uuid.UUID(batch_id)
        except ValueError:
            raise NotFound()

        batch_request = BatchRequest.objects.get(id=batch_id, type=self.batch_request_type)
        result = {
            'id': batch_request.id,
            'is_done': True,
            'requests': [],
        }

        role_requests = (
            batch_request.requests
            .values(
                'label', 'is_done', 'error',
                'role__id', 'role__is_active', 'role__state',
            )
        )

        for role_request in role_requests:
            role_request['role'] = {
                'id': role_request.pop('role__id'),
                'is_active': role_request.pop('role__is_active'),
                'state': role_request.pop('role__state'),
            }
            if role_request['role']['id'] is None:
                role_request['role'] = None
            result['is_done'] &= role_request['is_done']
            result['requests'].append(role_request)

        return self.create_response(request, result)

    def post_list(self, request, **kwargs):
        post_data = self.deserialize(request, request.body)
        requester = self.get_requester(request)

        requests = post_data.get('requests', [])
        if not isinstance(requests, list) or len(requests) == 0:
            raise BadRequest('`requests` property must be non-empty list of role requests')

        max_count = constance.config.MAX_BATCH_ROLEREQUESTS_COUNT
        if len(requests) > max_count:
            raise BadRequest(f'you can\'t request more than {max_count} roles in one batch request')

        batch_request = BatchRequest.objects.create(
            type=self.batch_request_type,
            requester=requester.as_dict(),
        )

        delayed_requests = self._prepare_delayed_requests(requester, batch_request, requests)
        DelayedRoleRequest.objects.bulk_create(delayed_requests)
        transaction.on_commit(
            lambda: ExecuteDelayedRoleRequests.delay(batch_request_id=str(batch_request.id)),
        )

        return self.create_response(request, {'id': batch_request.id}, status=201)

    def _prepare_delayed_requests(self, requester, batch_request, requests):
        raise NotImplementedError


class BatchRoleRequestResource(BaseBatchRequestResource):

    batch_request_type = BATCH_REQUEST_TYPE.GRANT

    class Meta(BaseBatchRequestResource.Meta):
        abstract = False
        resource_name = 'batchrolerequest'

    def _prepare_delayed_requests(self, requester, batch_request, requests):
        delayed_requests = []
        used_labels = set()

        for role_request in requests:
            delayed_request = DelayedRoleRequest(batch_request=batch_request)
            delayed_requests.append(delayed_request)

            form = DelayedRoleRequestForm(role_request)

            if not form.is_valid():
                delayed_request.error = form.get_error_message()
                delayed_request.is_done = True

            data = form.cleaned_data
            delayed_request.label = data.get('label')

            if not delayed_request.label or delayed_request.label in used_labels:
                raise BadRequest('every role request must have unique `label` property')
            used_labels.add(delayed_request.label)

            if delayed_request.is_done:
                continue

            is_public_role = data['path'].is_public
            is_permitted_for_private = data['system'].is_permitted_for(requester, 'core.request_role')

            if not is_public_role and not is_permitted_for_private:
                delayed_request.error = _('Вы не можете запрашивать скрытую роль')
                delayed_request.is_done = True
                continue

            delayed_request.data = role_request

        return delayed_requests


class BatchRoleDepriveResource(BaseBatchRequestResource):

    batch_request_type = BATCH_REQUEST_TYPE.DEPRIVE

    class Meta(BaseBatchRequestResource.Meta):
        abstract = False
        resource_name = 'batchroledeprive'

    def _prepare_delayed_requests(self, requester, batch_request, requests):
        delayed_requests = []
        used_roles = set()

        for request in requests:
            delayed_request = DelayedRoleRequest(batch_request=batch_request, data=request)
            delayed_requests.append(delayed_request)

            form = DelayedRoleDepriveForm(request)

            if not form.is_valid():
                delayed_request.error = form.get_error_message()
                delayed_request.is_done = True

            delayed_request.role = role = form.cleaned_data.get('id')
            if not role:
                raise BadRequest('every request must have `id` property with id of existing role')
            if role in used_roles:
                raise BadRequest('every request must have unique `id` property')
            used_roles.add(role)

        return delayed_requests
