# -*- coding: utf-8 -*-
import itertools
import logging
import operator
from collections import defaultdict
from typing import Dict, TypedDict, List, Any

import waffle
from django.conf import settings
from django.core.cache import cache
from django.core.handlers.wsgi import WSGIRequest
from django.db.models import Q, QuerySet, F

from idm.api.frontend.base import FrontendApiResource
from idm.core.models import System, RoleAlias, Role, RoleNode, GroupMembershipSystemRelation
from idm.users.constants.user import USER_TYPES
from idm.users.models import User

log = logging.getLogger(__name__)

FIREWALL_ROBOT_USERNAME = 'robot-dynfw-imports'
FIREWALL_RULES_CACHE_KEY = 'firewall:{user}:{system}:{expand_groups}:{include_personal}'


class LocalizedString(TypedDict):
    ru: str
    en: str


class MacroContent(TypedDict):
    users: List[str]
    groups: List[str]


class Macro(TypedDict):
    macro: str
    description: LocalizedString
    path: str
    content: MacroContent


class FirewallRule(TypedDict):
    name: LocalizedString
    slug: str
    macros: List[Macro]


def _relate_qs_filter_kwargs(relation_path: str, /, **kwargs) -> Dict[str, Any]:
    relation_path = relation_path.rstrip('_')
    return {'__'.join((relation_path, field)): value for field, value in kwargs.items()}


def get_cache_key(
    user: User,
    system_slug: str = None,
    expand_groups: bool = False,
    include_personal: bool = True,
) -> str:
    return FIREWALL_RULES_CACHE_KEY.format(
        user=user.username,
        system=system_slug or 'any',
        expand_groups=int(expand_groups),
        include_personal=int(include_personal),
    )


def get_firewall_rules(
    user: User,
    system_slug: str = None,
    expand_groups: bool = False,
    include_personal: bool = True,
) -> List[FirewallRule]:
    system_filter = {'is_active': True}
    if system_slug:
        system_filter.update(slug=system_slug)

    # получаем все активные firewall-алиасы подгружая название системы
    firewall_aliases: QuerySet[RoleAlias] = (
        RoleAlias.objects.get_firewalls()
            .select_related(None)
            .select_related('node')
            .annotate(system_slug=F('node__system__slug'))
            .filter(is_active=True, **_relate_qs_filter_kwargs('node__system', **system_filter))
            .order_by()
    )

    alias_by_node = defaultdict(set)
    system_aliases: Dict[str, List[RoleAlias]] = defaultdict(list)
    for alias in firewall_aliases:
        alias_by_node[alias.node_id].add(alias)
        system_aliases[alias.system_slug].append(alias)

    node_subtree = RoleNode.objects.active().filter(
        is_key=False,
        **_relate_qs_filter_kwargs('system', **system_filter),
        rolenodeclosure_parents__parent_id__in=firewall_aliases.values_list('node_id', flat=True)
    ).annotate(subtree_head_id=F('rolenodeclosure_parents__parent_id')).order_by()

    subtree_heads_map = defaultdict(set)
    for node_id, subtree_head_id in node_subtree.values_list('id', 'rolenodeclosure_parents__parent_id'):
        subtree_heads_map[node_id].add(subtree_head_id)

    is_group_role = Q(group__isnull=False)
    is_personal_role = Q(
        Q(parent__isnull=False) & Q(parent__group__isnull=False),
        user__isnull=False, user__type=USER_TYPES.USER
    )
    is_user_role = Q(
        Q(parent__isnull=True) | Q(parent__group__isnull=True),
        user__isnull=False, user__type=USER_TYPES.USER
    )
    subject_q = is_user_role
    subject_fields = ['user__username']
    if not expand_groups:
        subject_q |= is_group_role
        subject_fields.append('group__external_id')
    elif include_personal:
        subject_q |= is_personal_role

    roles_qs = Role.objects.active().filter(subject_q, **_relate_qs_filter_kwargs('system', **system_filter)).order_by()
    if user.username != FIREWALL_ROBOT_USERNAME:
        roles_qs = roles_qs.permitted_for(user)

    system_roles_querysets = [roles_qs.values_list('system__slug', *subject_fields)]
    if expand_groups:
        system_roles_querysets.append(
            GroupMembershipSystemRelation.objects.active().values_list('system__slug', 'membership__user__username')
        )

    group_all_roles = defaultdict(set)
    user_all_roles = defaultdict(set)
    for system_slug, *subject_values in itertools.chain(*system_roles_querysets):
        if not expand_groups:
            username, group_id = subject_values
        else:
            username, *_ = subject_values
            group_id = None

        if group_id is not None:
            group_all_roles[system_slug].add(group_id)
        elif username is not None:
            user_all_roles[system_slug].add(username)
        else:
            log.error(f'System {system_slug} has no subject at all roles firewall aggregation')
            raise ValueError('Отсутствует владелец роли в рамках агрегации правил фаервола')  # для проверки

    # system_slug -> alias_name -> usernames | group_ids
    groups_by_macro = defaultdict(lambda: defaultdict(set))
    usernames_by_macro = defaultdict(lambda: defaultdict(set))
    for system_slug, node_id, *subject_values in (
        roles_qs.filter(node__in=node_subtree).values_list(
            'system__slug',
            'node_id',
            *subject_fields,
        )
    ):
        if not expand_groups:
            username, group_id = subject_values
        else:
            username, *_ = subject_values
            group_id = None
            assert not _

        for subtree_head_id in subtree_heads_map[node_id]:
            for macro in alias_by_node[subtree_head_id]:
                if group_id is not None:
                    groups_by_macro[system_slug][macro.name_en].add(group_id)
                elif username is not None:
                    usernames_by_macro[system_slug][macro.name_en].add(username)
                else:
                    log.error(f'System {system_slug} has no subject at macro "{macro}" firewall aggregation')
                    raise ValueError('Отсутствует владелец роли в рамках агрегации правил фаервола')  # для проверки

    firewall_rules = []
    for system_slug, system_name, system_name_en, system_root_id in (
        System.objects.filter(**system_filter).values_list('slug', 'name', 'name_en', 'root_role_node_id')
    ):
        macro = {
            f'system_{system_slug}': Macro(
                macro=f'system_{system_slug}',
                description=LocalizedString(
                    ru=f'{system_name}: Все роли',
                    en=f'{system_name_en}: All roles',
                ),
                path='/',
                content=MacroContent(
                    users=sorted(user_all_roles.get(system_slug, [])),
                    groups=sorted(group_all_roles.get(system_slug, []))
                )
            )
        }
        for alias in system_aliases.get(system_slug, []):  # type: RoleAlias
            if alias.node.parent_id == system_root_id:
                path = '/'
                description_ru, description_en = 'Все роли', 'All roles'
            else:
                path = alias.node.value_path
                description_ru = alias.node.humanize(lang='ru', format='short')
                description_en = alias.node.humanize(lang='en', format='short')
            macro[f'{system_slug}_{alias.name_en}'] = Macro(
                macro=f'{system_slug}_{alias.name_en}',
                description=LocalizedString(
                    ru=f'{system_name}: {description_ru}',
                    en=f'{system_name_en}: {description_en}'),
                path=path,
                content=MacroContent(users=[], groups=[]),
            )

        for alias_name_en, usernames in usernames_by_macro.get(system_slug, {}).items():
            macro[f'{system_slug}_{alias_name_en}']['content']['users'].extend(usernames)
            macro[f'{system_slug}_{alias_name_en}']['content']['users'].sort()

        if not expand_groups:
            for alias_name_en, group_ids in groups_by_macro.get(system_slug, {}).items():
                macro[f'{system_slug}_{alias_name_en}']['content']['groups'].extend(group_ids)
                macro[f'{system_slug}_{alias_name_en}']['content']['groups'].sort()

        firewall_rules.append({
            'name': LocalizedString(ru=system_name, en=system_name_en),
            'slug': system_slug,
            'macros': sorted(macro.values(), key=operator.itemgetter('macro')),
        })
    return firewall_rules


class FirewallResource(FrontendApiResource):
    CACHE_DEFAULT_SWITCH = 'firewall_rules_cache_default'

    class Meta:
        object_class = None
        resource_name = 'firewall/rules'
        list_allowed_methods = ['get']
        detail_allowed_methods = []

    def get_list(self, request: WSGIRequest, **kwargs):
        user: User = request.user  # noqa
        system_slug = request.GET.get('system')
        expand_groups = request.GET.get('allow_groups', 'false').lower() in ('false', '0')
        use_cache = request.GET.get(
            'use_cache', str(waffle.switch_is_active(self.CACHE_DEFAULT_SWITCH)).lower()
        ).lower() not in ('false', '0')

        cache_key = get_cache_key(user, system_slug, expand_groups=expand_groups)
        if not use_cache or not (result := cache.get(cache_key, [])):
            result = get_firewall_rules(user, system_slug=system_slug, expand_groups=expand_groups)
            cache.set(cache_key, result, timeout=settings.FIREWALL_RULES_CACHE_TIMEOUT)

        return self.create_response(request, result)
