from django.db import connection, models
from django.db.models import F
from psycopg2.extras import _paginate


def execute_batch(cur, sql, argslist, page_size=100):
    """
    Переписанный psycopg2.extras.execute_batch.
    В отличие от оригинала работает со строками вместо байтов.
    Используем его, т.к. django_tools_log_context с байтами падает.
    """
    for page in _paginate(argslist, page_size=page_size):
        sqls = (cur.mogrify(sql, args) for args in page)
        sqls = (sql.decode('utf-8') if hasattr(sql, 'decode') else sql for sql in sqls)
        cur.execute(';'.join(sqls))


class CttManager(models.Manager):

    INSERT_SQL_TEMPLATE = (
        'INSERT INTO {table} (depth, child_id, parent_id) '
        'SELECT 0, %s, %s '
        'UNION SELECT depth + 1, %s, parent_id '
        'FROM {table} WHERE child_id = %s '
        'ON CONFLICT DO NOTHING'
    )

    @property
    def insert_sql(self):
        table = self.model._closure_model._meta.db_table
        return self.INSERT_SQL_TEMPLATE.format(table=table)

    def bulk_create(self, *args, **kwargs):
        if not kwargs.pop('ctt_considered', False):
            raise RuntimeError(
                "Please don't forget to call closure_createlink() on model's "
                "manager for each created row and call bulk_create() "
                "with ctt_considered=True")

        return super(CttManager, self).bulk_create(*args, **kwargs)

    def closure_createlink(self, pk, parent_id):
        query_args = [pk, pk, pk, parent_id]
        with connection.cursor() as cursor:
            cursor.execute(self.insert_sql, query_args)

    def closure_bulk_createlink(self, nodes):
        args = ([n.pk, n.pk, n.pk, n._closure_parent_pk] for n in nodes)
        with connection.cursor() as cursor:
            execute_batch(cursor, self.insert_sql, args)

    def closure_update_links(self, instance, new_parent, old_parent):
        subtree_with_self = instance.get_descendants(include_self=True)
        subtree_without_self = instance.get_descendants().order_by('level')
        cached_subtree = [instance] + list(subtree_without_self)
        new_level = new_parent.level if new_parent is not None else 0
        old_level = old_parent.level if old_parent is not None else 0
        leveldiff = new_level - old_level
        if leveldiff != 0:
            subtree_without_self.update(level=F('level') + leveldiff)
        links = self.model._closure_model.objects.filter(child_id__in=subtree_with_self)
        links.delete()
        self.closure_bulk_createlink(cached_subtree)
