from constance import config
from dbtemplates.models import Template
from django.conf import settings
from django.conf.urls import url
from django.contrib import messages
from django.contrib.admin import ACTION_CHECKBOX_NAME
from django.contrib.admin import (
    AdminSite,
    ModelAdmin,
    StackedInline,
    TabularInline,
    SimpleListFilter,
    site,
)
from django.contrib.auth.decorators import permission_required
from django.template.response import TemplateResponse
from django.urls import reverse
from django.utils.decorators import method_decorator
from django.utils.encoding import force_text
from django.utils.html import format_html
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy as _
from django.views.generic import TemplateView
from tastypie.models import ApiKey

from idm.core import depriving
from idm.core import models
from idm.core.constants.instrasearch import INTRASEARCH_METHOD
from idm.core.constants.role import ROLE_STATE
from idm.core.depriving import get_depriving_roles_check_by_name
from idm.core.tasks.roles import DepriveDeprivingRoles
from idm.framework.base_admin import IdmModelAdminMixin, DefaultIdmModelAdmin
from idm.services.models import Service


class SystemRoleFieldsInline(IdmModelAdminMixin, StackedInline):
    model = models.SystemRoleField
    extra = 0


class SystemAdmin(IdmModelAdminMixin, ModelAdmin):
    """
    Подключенные системы
    """
    inlines = [SystemRoleFieldsInline]
    list_display = (
        'slug', 'name', 'name_en', 'added', 'is_sox', 'is_active', 'is_broken', 'use_requests',
    )
    list_filter = (
        'is_active', 'is_broken', 'is_sox', 'has_review',
        'passport_policy', 'group_policy', 'request_policy',
        'role_grant_policy', 'roletree_policy', 'review_on_relocate_policy',
        'inconsistency_policy', 'workflow_approve_policy',
        'audit_method', 'can_be_broken'
    )
    list_per_page = 200
    raw_id_fields = ('actual_workflow', 'root_role_node', 'service', 'creator')
    search_fields = ('slug', 'name', 'name_en')

    def get_queryset(self, request):
        return models.System.objects.select_related('metainfo')

    def save_model(self, request, obj, form, change):
        if obj.auth_factor == 'tvm' and not obj.tvm_id:
            raise ValueError('Can\'t set auth factor to tvm when tvm_id is not provided')
        obj.slug = obj.slug.lower()
        obj.save()


class FavoriteSystemAdmin(IdmModelAdminMixin, ModelAdmin):
    """
    Избранные системы
    """
    list_display = ('user', 'system')
    search_fields = ('user__username',)
    raw_id_fields = ('user', 'system')


class WorkflowAdmin(IdmModelAdminMixin, ModelAdmin):
    """
    Данные по wf
    """
    editable = False
    list_display = (
        'system', 'user', 'state', 'updated',
    )
    search_fields = (
        'system__name',
        'system__slug'
    )
    ordering = ('-updated',)
    list_filter = ('state', 'system')


class ServiceAdmin(IdmModelAdminMixin, ModelAdmin):
    editable = False
    list_display = (
        'name', 'external_state', 'is_vteam', 'updated_at', 'root'
    )
    search_fields = (
        'slug',
        'name',
        'name_en',
    )
    ordering = ('-updated_at',)
    list_filter = ('is_vteam', 'external_state')


class RoleAdmin(IdmModelAdminMixin, ModelAdmin):
    """
    Данные по ролям из поключенных систем
    """
    editable = True
    list_display = (
        'system', 'user', 'node', 'state', 'fields_data', 'system_specific', 'is_active', 'added', 'updated',
    )

    search_fields = (
        'system__name',
        'system__slug',
        'user__username',
        'node__value_path',
        'fields_data',
        'system_specific',
    )

    list_filter = ('system__name', 'state')
    raw_id_fields = (
        'user',
        'group',
        'inconsistency',
        'parent',
        'node',
        'last_request',
    )
    actions = (
        'deprive_depriving_roles',
    )

    def deprive_depriving_roles(self, request, queryset):
        """
        Отзывает из выбранных ролей те, что находятся в статусе "валидация отзыва"
        """
        queryset = queryset.filter(state=ROLE_STATE.DEPRIVING_VALIDATION)
        count = queryset.count()
        if not count:
            messages.warning(request, 'Не выбрано ни одной роли в статусе "валидация отзыва"')
            return self._redirect_back(request)
        if request.POST.get('is_confirmed'):
            roles_ids = list(queryset.values_list('id', flat=True))
            DepriveDeprivingRoles.delay(depriver_id=None, roles_ids=roles_ids, block=True)
            messages.info(request, 'Роли поставились в очередь на отзыв')
            return self._redirect_back(request)

        context = {
            'action_checkbox_name': ACTION_CHECKBOX_NAME,
            'media': self.media,
            'queryset': queryset,
            'count': count,
            'is_checked': False,
            'referer': request.META.get('HTTP_REFERER'),
        }

        if request.POST.get('check'):
            check = get_depriving_roles_check_by_name(request.POST['check'])
            context['check_description'] = check.__doc__
            context['check_errors'] = check(queryset, request.POST)
            context['is_checked'] = True

        return TemplateResponse(request, 'idm_admin/depriving_validation_confirm.html', context)

    def has_validate_depriving_roles_permission(self, request, obj=None):
        return request.user.has_perm('core.idm_validate_depriving_roles')

    deprive_depriving_roles.short_description = 'Отозвать роли на валидации отзыва'
    deprive_depriving_roles.allowed_permissions = ('validate_depriving_roles',)


class RoleFieldsInline(IdmModelAdminMixin, StackedInline):
    model = models.RoleField
    extra = 0


class RoleAliasesInline(IdmModelAdminMixin, StackedInline):
    model = models.RoleAlias
    extra = 0


class RoleResponsibilitiesInline(IdmModelAdminMixin, TabularInline):
    model = models.NodeResponsibility
    raw_id_fields = ('user',)
    extra = 0


class RoleNodeAdmin(IdmModelAdminMixin, ModelAdmin):
    editable = True
    inlines = [RoleFieldsInline, RoleAliasesInline, RoleResponsibilitiesInline]
    list_display = (
        'humanize',
        'system',
        'level',
        'is_public',
        'is_auto_updated',
        'is_key',
    )
    list_filter = (
        'state',
        'is_public',
        'is_auto_updated',
        'is_key',
        'system',
    )
    list_editable = ('is_auto_updated',)
    list_select_related = ('system',)
    search_fields = ('name', 'name_en')
    ordering = ('system_id', 'level', 'id')
    raw_id_fields = ('parent', 'moved_from')

    def save_model(self, request, obj, form, change):
        if obj.is_public and obj.is_active():
            method = INTRASEARCH_METHOD.ADD
        else:
            method = INTRASEARCH_METHOD.REMOVE
        obj.save()
        obj.send_intrasearch_push(method)

    def humanize(self, obj):
        if obj.is_root_node():
            return '[Корневой узел]'
        else:
            return obj.humanize()
    humanize.short_description = 'Описание'


class RoleNodeSetAdmin(IdmModelAdminMixin, ModelAdmin):
    list_display = ('set_id', 'is_active', 'system')
    list_filter = ('is_active', 'system')


class PassportLoginAdmin(DefaultIdmModelAdmin):
    """
    Паспортные логины
    """
    list_display = ('login', 'user', 'state', 'added')
    search_fields = ('user__username', 'login')
    raw_id_fields = ('roles', 'user')

    def get_queryset(self, request):
        qs = super(PassportLoginAdmin, self).get_queryset(request)
        qs = qs.select_related('user')
        return qs


class ActionAdmin(IdmModelAdminMixin, ModelAdmin):
    """Данные по действиям"""
    def reencoded_data(self, action):
        result = _('(Ничего)')
        if action.data:
            try:
                result = force_text(action.data).decode('unicode_escape')
            except UnicodeDecodeError:
                result = action.data
        return result
    reencoded_data.short_description = 'Данные'

    def inconsistency_link(self, action):
        result = ''
        if action.inconsistency_id:
            url = reverse('admin:idm_inconsistency_change', args=(action.inconsistency_id, ))
            result = mark_safe('<a href="{url:s}">[Расхождение]</a>'.format(url=url))
        return result

    inconsistency_link.short_description = 'Расхождение'

    editable = False
    list_display = (
        'added',
        'user',
        'action',
        'reencoded_data',
        'inconsistency_link',
        'system',
    )

    search_fields = (
        'group__slug',
        'user__username',
        'role__node__value_path',
    )

    list_filter = ('action', 'system', )
    date_hierarchy = 'added'
    raw_id_fields = (
        'role', 'user', 'inconsistency', 'group', 'rolerequest', 'approverequest', 'role_node',
        'role_field', 'role_alias', 'requester', 'impersonator', 'membership', 'workflow', 'responsibility',
        'parent',
    )


class TransferAdmin(IdmModelAdminMixin, ModelAdmin):
    """Перемещения"""

    editable = False
    list_display = ('get_owner', 'type', 'other', 'state', 'get_source', 'get_target', 'created_at', 'resolved_at')
    search_fields = ('user__username', 'group__name', 'group__slug', 'group__external_id')
    list_filter = ('state', 'type', 'is_new')
    actions = ['accept', 'reject']
    raw_id_fields = ('user', 'group')
    date_hierarchy = 'created_at'

    # admin actions
    def accept(self, request, queryset):
        from idm.core.tasks.nodes import ResolveTransfersTask
        ResolveTransfersTask.apply_async(
            kwargs={
                'pks': list(queryset.values_list('pk', flat=True)),
                'decision': 'accept',
                'requester_username': request.user.username,
            }
        )
    accept.short_description = _('Пересмотреть')

    def reject(self, request, queryset):
        from idm.core.tasks.nodes import ResolveTransfersTask
        ResolveTransfersTask.apply_async(
            kwargs={
                'pks': list(queryset.values_list('pk', flat=True)),
                'decision': 'reject',
                'requester_username': request.user.username,
            }
        )

    reject.short_description = _('Оставить')

    # admin properties
    def get_owner(self, obj):
        return obj.owner
    get_owner.short_description = _('Субъект перемещения')

    def get_source(self, obj):
        return obj.get_source()
    get_source.short_description = _('Откуда')

    def get_target(self, obj):
        return obj.get_target()
    get_target.short_description = _('Куда')

    def other(self, obj):
        result = '[Нет]'
        if obj.type == 'user_group' and obj.parent:
            group_url = reverse('admin:core_transfer_changelist') + '?id=%s' % obj.parent.pk
            url = reverse('admin:core_transfer_changelist') + '?parent=%s' % obj.parent.pk
            result = format_html('<a href="{group_url}">[Перемещение&nbsp;группы]</a><br>'
                                 '<a href="{url}">[Перемещения людей]</a>', group_url=group_url, url=url)
        elif obj.type == 'group':
            url = reverse('admin:core_transfer_changelist') + '?parent=%s' % obj.pk
            result = format_html('<a href="{url}">[Перемещения&nbsp;людей]</a>', url=url)
        return result
    other.short_description = _('Связанные')


class DelayedRoleRequestInline(StackedInline):

    model = models.DelayedRoleRequest
    extra = 0
    fields = (
        'label',
        'is_done',
        'data',
        'error',
        'get_role',
    )
    readonly_fields = fields

    def get_role(self, obj):
        idm_url = settings.IDM_BASE_URL
        role_prefix = '/reports/roles#f-is-expanded=true,f-status=all,f-system-type=all,f-role-id='
        if obj.role_id:
            return mark_safe(f'<a href="{idm_url}{role_prefix}{obj.role_id}">{obj.role_id}</a>')
        return None
    get_role.short_description = 'Role'


class IsDoneRequestListFilter(SimpleListFilter):

    title = 'Is Done'
    parameter_name = 'is_done'

    def lookups(self, request, model_admin):
        return (
            ('True', 'True'),
            ('False', 'False'),
        )

    def queryset(self, request, queryset):
        not_finished = queryset.filter(requests__is_done=False).distinct()
        if self.value() == 'True':
            return queryset.exclude(id__in=not_finished)
        elif self.value() == 'False':
            return not_finished
        return queryset


class BatchRequestAdmin(IdmModelAdminMixin, ModelAdmin):

    fields = (
        'id',
        'type',
        'requester',
        'get_is_done',
    )
    readonly_fields = fields
    list_display = fields
    inlines = (DelayedRoleRequestInline,)
    list_filter = (IsDoneRequestListFilter,)

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.prefetch_related('requests')

    def get_is_done(self, obj):
        for request in obj.requests.all():
            if not request.is_done:
                return False
        return True


site.register(models.UserPassportLogin, PassportLoginAdmin)
site.register(models.Action, ActionAdmin)
site.register(models.System, SystemAdmin)
site.register(models.FavoriteSystem, FavoriteSystemAdmin)
site.register(models.Role, RoleAdmin)
site.register(models.RoleNode, RoleNodeAdmin)
site.register(models.RoleNodeSet, RoleNodeSetAdmin)
site.register(models.Transfer, TransferAdmin)
site.register(models.Workflow, WorkflowAdmin)
site.register(Service, ServiceAdmin)
site.register(models.BatchRequest, BatchRequestAdmin)

site.unregister(ApiKey)
site.register(
    ApiKey,
    DefaultIdmModelAdmin,
    list_display=(
        'user',
        'key',
        'created',
    ),
    search_fields=('user__username', 'key'),
)

site.unregister(Template)
site.register(Template, DefaultIdmModelAdmin)

from django.contrib.auth.models import Permission
site.register(Permission)

# убираем из админки ненужное
site.disable_action('delete_selected')  # удалять ничего в админке нельзя


class IdmAdminSite(AdminSite):
    index_template = 'idm_admin/index.html'

    def has_permission(self, request):
        # Даем юзерам с пермишеном view_matrix доступ к oebs-conflicts
        if request.path == reverse('admin:oebs-conflicts') and request.user.has_perm('core.idm_view_matrix'):
            return True
        return request.user.is_active and request.user.is_staff

    def get_urls(self):
        from idm.core.oebs_conflicts import upload_oebs_conflicts

        urls = super(IdmAdminSite, self).get_urls()
        idm_urls = [
            url(r'^oebs-conflicts/$', self.admin_view(upload_oebs_conflicts), name='oebs-conflicts'),
            url(
                r'^depriving-validation/$',
                self.admin_view(DeprivingValidationAdminView.as_view()),
                name='depriving-validation',
            ),
        ]
        idm_urls += urls
        return idm_urls


@method_decorator(permission_required('core.idm_validate_depriving_roles'), 'dispatch')
class DeprivingValidationAdminView(TemplateView):
    """
    Страница в админке для валидации отзыва ролей
    """
    template_name = 'idm_admin/depriving_validation.html'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        roles = depriving.get_depriving_roles()
        return dict(
            context,
            roles=roles,
            grouped_roles=depriving.group_roles(roles),
            action_checkbox_name=ACTION_CHECKBOX_NAME,
            idm_url=settings.IDM_BASE_URL,
            max_roles_count=config.ADMIN_MAX_DEPRIVE_ROLES_COUNT,
        )


idm_site = IdmAdminSite()
