# coding: utf-8
import enum
from typing import Dict, Tuple, Iterable

import attr
from django.db.models import Q, F
from django.utils.translation import ugettext_lazy as _


@attr.s
class APPROVEREQUEST_DECISION(object):
    APPROVE = 'approve'
    DECLINE = 'decline'
    IGNORE = 'ignore'
    DISCUSS = 'discuss'
    NOT_DECIDED = ''

    ABBREVIATION = {
        APPROVE: 'Y',
        DECLINE: 'N',
        IGNORE: 'I',
        DISCUSS: 'D',
        NOT_DECIDED: 'W',
    }

    TRANSLATIONS = {
        APPROVE: _('Подтвердить'),
        DECLINE: _('Отклонить'),
        IGNORE: _('Игнорировать'),
        DISCUSS: _('Обсудить'),
        NOT_DECIDED: '',
    }


class PriorityType(enum.Enum):
    _ignore_ = ('_TRANSLATIONS', '_QUERYSET_FILTERS')
    PRIORITY_PRIMARY = 'primary'
    PRIORITY_SECONDARY = 'secondary'
    PRIORITY_ALL = 'all'

    _TRANSLATIONS: Dict['PriorityType', str]
    _QUERYSET_FILTERS: Dict['PriorityType', str]

    @property
    def queryset_filter(self):
        return self._QUERYSET_FILTERS[self]

    @classmethod
    def choices(cls) -> Iterable[Tuple[str, str]]:
        return tuple((item.value, translation) for item, translation in cls._TRANSLATIONS.items())


PriorityType._TRANSLATIONS = {
    PriorityType.PRIORITY_PRIMARY: _('Основной'),
    PriorityType.PRIORITY_SECONDARY: _('Второстепенный'),
    PriorityType.PRIORITY_ALL: _('Все'),
}

PriorityType._QUERYSET_FILTERS = {
    PriorityType.PRIORITY_ALL: Q(),
    PriorityType.PRIORITY_PRIMARY: Q(priority__lte=F('approve__main_priority')),
    PriorityType.PRIORITY_SECONDARY: Q(priority__gt=F('approve__main_priority')),
}
