from collections import defaultdict, namedtuple
from itertools import chain
from typing import Optional, Dict, Any, Union, Set, List, Callable, TypedDict, Collection

from django.conf import settings
from django.db.models import Q, F, Case, When, Value, CharField, QuerySet, prefetch_related_objects
from django.utils import timezone
from ids.registry import registry

from idm.core.constants.action import ACTION, ACTION_STATE
from idm.core.constants.role import ROLE_STATE
from idm.core.models import Action, System, Role
from idm.monitorings.metric import FailedDeprivingRolesMetric, TotalDeprivingRolesMetric
from idm.users.constants.group import GROUP_TYPES
from idm.users.models import User, Group
from idm.users.utils import get_group_url
from idm.utils import chunkify


def cache_failed_roles_count(failed_count: int):
    FailedDeprivingRolesMetric.set(failed_count)


def cache_total_roles_count(depriving_count: int):
    TotalDeprivingRolesMetric.set(depriving_count)


def get_roles_data(system: System = None) -> QuerySet[Dict[str, Any]]:
    filter_query = Q(
        action__in=(ACTION.DEPRIVE, ACTION.EXPIRE),
        role__state=ROLE_STATE.DEPRIVING_VALIDATION,
        role__depriving_at__lte=timezone.now(),
        role__system__is_active=True,
        role__system__is_broken=False,
    )
    roles_with_parent = (
        # Те роли, у которых родительская активна (кроме DEPRIVING_VALIDATION)
            Q(role__parent__state__in=ROLE_STATE.ACTIVE_RETURNABLE_STATES - {ROLE_STATE.DEPRIVING_VALIDATION})
            & (
                # Пользователь не уволен
                    Q(role__user__is_active=True)
                    # Или роль выдана на группу
                    | Q(role__user__isnull=True)
            )
    )

    filter_query &= (Q(role__parent__isnull=True) | roles_with_parent)

    action_state_query = (
            ~Q(parent__data__action_state=ACTION_STATE.BEGIN)
            | Q(parent__data__action_state__isnull=True)
    )

    if system:
        filter_query &= Q(role__system_id=system.id)

    filter_query &= action_state_query

    # Уберем дефолтную сортировку, чтоб сделать запрос чуть быстрее
    return Action.objects.filter(filter_query).values('role', 'role__depriver', 'role__user', 'role__group').order_by()


def group_role_data_by_depriver(system: System = None) -> Dict[Optional[User], Dict[str, Set[Union[str, int]]]]:
    role_data_by_depriver = defaultdict(lambda: defaultdict(set))

    for item in get_roles_data(system).iterator():
        depriver: User = item['role__depriver']
        role_id: Role = item['role']
        subject = f'user_{item["role__user"]}' if item['role__user'] is not None else f'group_{item["role__group"]}'
        role_data_by_depriver[depriver]['subjects'].add(subject)
        role_data_by_depriver[depriver]['roles'].add(role_id)

    return role_data_by_depriver


def get_roles_ids_to_deprive(system: System = None, force: bool = False) -> Dict[Optional[User], Set[int]]:
    role_ids_by_depriver = {}

    for depriver, role_data in group_role_data_by_depriver(system).items():
        role_ids_by_depriver[depriver] = role_data['roles']

    return role_ids_by_depriver


class DeprivingRoleDict(TypedDict):
    id: int
    user_id: str
    group_id: str
    parent_id: int
    parent_group_id: str
    parent_group_type: str
    parent_group_slug: str
    parent_group_parent_slug: str
    system_slug: str
    ownership: str


def get_depriving_roles() -> List[DeprivingRoleDict]:
    return list(
        Role.objects
            .filter(id__in=group_role_data_by_depriver()[None]['roles'])
            .values(
            'id',
            'user_id',
            'group_id',
            'parent_id',
            parent_group_id=F('parent__group_id'),
            parent_group_type=F('parent__group__type'),
            parent_group_slug=F('parent__group__slug'),
            parent_group_parent_slug=F('parent__group__parent__slug'),
            system_slug=F('system__slug'),
            ownership=Case(
                When(user__isnull=False, then=Value('personal')),
                When(group__isnull=False, then=Value('group')),
                default=Value('unknown'),
                output_field=CharField(),
            ),
        )
    )


LiteGroupBase = namedtuple('LiteGroupBase', ['id', 'type', 'slug', 'parent_slug'])


class LiteGroup(LiteGroupBase):

    @property
    def url(self):
        if not self.type:
            return None
        elif self.type == GROUP_TYPES.SERVICE:
            # Note: легкий костыльный способ получить слаг сервиса для servicerole-групп
            slug = self.parent_slug if self.slug.startswith(self.parent_slug) else self.slug
            return settings.IDM_GROUP_LINK_TEMPLATE_ABC % dict(slug=slug)
        return get_group_url(self)


class RoleGrouping(list):

    @property
    def roles_count(self):
        return len(self)

    @property
    def subjects_count(self):
        user_ids = {r['user_id'] for r in self if r.get('user_id')}
        group_ids = {r['group_id'] for r in self if r.get('group_id')}
        return len(user_ids) + len(group_ids)


def group_roles(roles: List[DeprivingRoleDict]) -> Dict[str, Dict[str, RoleGrouping]]:
    grouping_keys = (
        'parent_id',
        'parent_group',
        'system_slug',
        'ownership',
    )
    result = defaultdict(dict)
    for role in roles:
        if role['parent_group_id']:
            role['parent_group'] = LiteGroup(
                id=role['parent_group_id'],
                type=role['parent_group_type'],
                slug=role['parent_group_slug'],
                parent_slug=role['parent_group_parent_slug'],
            )
        for key in grouping_keys:
            value = role.get(key)
            if value:
                grouping = result[f'by_{key}'].setdefault(value, RoleGrouping())
                grouping.append(role)
    return result


def get_depriving_roles_check_by_name(check_name: str) -> Callable:
    return globals().get(f'check_depriving_roles_by_{check_name}')


def check_depriving_roles_by_group(roles: QuerySet[Role], data: Dict[str, Any]) -> List[str]:
    """
    Проверка на то, что обладатели заданных ролей больше не входят в заданную группу
    """
    assert data.get('slug')
    group = Group.objects.get(slug=data['slug'])
    prefetch_related_objects(roles, 'user')
    errors = []

    non_user_role_ids = []
    usernames = set()
    for role in roles:
        if not role.user:
            non_user_role_ids.append(role)
            continue
        usernames.add(role.user.username)

    if not usernames:
        errors.append(f'Среди переданных ролей нет ни одной персональной')
    elif non_user_role_ids:
        errors.append(f'Среди переданных ролей есть не персональные: {non_user_role_ids}')

    if usernames:
        include_subgroups = group.type == GROUP_TYPES.DEPARTMENT

        members = list(chain.from_iterable(
            _check_members_in_staff_groups(group.slug, chunk, include_subgroups)
            for chunk in chunkify(usernames, 100)
        ))
        if members:
            errors.append(f'Найдены люди, входящие в заданную группу: {members}')

    return errors


def _check_members_in_staff_groups(
        group_name: str,
        usernames: Collection[str],
        include_subgroups: bool = False,
) -> List[str]:
    assert usernames
    staff_repository = registry.get_repository(
        service='staff',
        resource_type='groupmembership',
        oauth_token=settings.IDM_STAFF_OAUTH_TOKEN,
        user_agent=settings.IDM_IDS_USER_AGENT,
        timeout=settings.IDM_IDS_TIMEOUT,
    )
    common_params = {
        'person.login': ','.join(usernames),
        '_limit': 1000,
        '_fields': 'person.login',
        '_sort': 'id',
    }
    members = []
    members.extend(staff_repository.getiter({'group.url': group_name, **common_params}))
    if include_subgroups:
        members.extend(staff_repository.getiter({'group.ancestors.url': group_name, **common_params}))

    return [member_data['person']['login'] for member_data in members]
