# coding: utf-8


import logging
from typing import Dict, Any, Generator, Union, List

import six

from idm.core.canonical import CanonicalNode, CanonicalAlias, CanonicalField, CanonicalResponsibility
from idm.users.models import User
from idm.utils.i18n import get_lang_pair

log = logging.getLogger(__name__)


class RoleNodeFetcher:
    user_cache: Dict[str, User]

    def __init__(self):
        self.user_cache = {}

    def prepare(self):
        self.user_cache = {user.username: user for user in User.objects.active().only('username')}

    def fetch(self, system) -> CanonicalNode:
        try:
            return self.normalize_tree(system.node_plugin.get_info())
        except Exception:
            log.warning('Cannot fetch role tree for system %s', system.slug, exc_info=True)
            raise

    def normalize_tree(self, structure: Dict[str, Any]) -> CanonicalNode:
        """
        Приводит в дереве ролей сокращенный формат описания роли {ключ: название} в подробный,
        а также приводит дерево в формат списков списков из словаря словарей
        """
        root = CanonicalNode(hash='', slug='')
        if 'roles' not in structure:
            return root
        global_fields = structure.get('fields')
        if global_fields:
            structure['roles']['fields'] = global_fields
        global_firewall = structure.get('firewall-declaration')
        if global_firewall:
            structure['roles']['firewall-declaration'] = global_firewall
        result = self.normalize_inner(structure['roles'], direct=True)
        if result:
            root.children = result
        return root

    @staticmethod
    def unify_children(
            item: Union[Dict[str, Any], List[Dict[str, Any]]],
            direct: bool,
    ) -> Generator[Dict[str, Any], None, None]:
        if direct:
            if isinstance(item, list):
                yield from item
            else:
                yield item
        else:
            for slug, name_or_dict in item.items():
                role_data = {'slug': slug}
                if isinstance(name_or_dict, six.string_types):
                    role_data['name'] = name_or_dict
                else:
                    role_data.update(name_or_dict)
                yield role_data

    def normalize_inner(self, item: Dict[str, Any], direct: bool):
        from idm.core.models import RoleAlias, RoleField
        canonicals = []
        for role_data in self.unify_children(item, direct):
            fields = role_data.get('fields', ())
            node_fields = {}
            for field_data in fields:
                field_type = field_data.get('type')
                if field_type is None:
                    if field_data['slug'] == 'passport-login':
                        field_type = 'passportlogin'
                    else:
                        field_type = 'charfield'
                options = field_data.get('options') or None
                RoleField.check_options(field_type, options)
                if options is not None:
                    choices = options.get('choices')
                    if choices is not None:
                        options['choices'] = RoleField.prepare_choices(choices)
                name, name_en = get_lang_pair(field_data.get('name', ''))
                dependencies = field_data.get('depends_on') or None
                canonical_field = CanonicalField(
                    type=field_type,
                    slug=field_data['slug'],
                    name=name,
                    name_en=name_en,
                    is_required=self.as_bool(field_data.get('required', False)),
                    options=options,
                    dependencies=dependencies
                )
                node_fields[canonical_field.as_key()] = canonical_field
            aliases = role_data.get('aliases', ())
            firewall = role_data.get('firewall-declaration', None)
            node_aliases = {}
            for alias in aliases:
                name, name_en = get_lang_pair(alias.get('name', ''))
                alias = CanonicalAlias(
                    type=alias.get('type', RoleAlias.DEFAULT_ALIAS),
                    name=name,
                    name_en=name_en
                )
                node_aliases[alias.as_key()] = alias
            if firewall:
                alias = CanonicalAlias(
                    type=RoleAlias.FIREWALL_ALIAS,
                    name=firewall,
                    name_en=firewall,
                )
                node_aliases[alias.as_key()] = alias
            responsibilities = role_data.get('responsibilities', ())
            node_responsibilities = {}
            if responsibilities:
                for responsibility in responsibilities:
                    user = self.user_cache.get(responsibility['username'])
                    if user is not None:
                        canonical = CanonicalResponsibility(user=user, notify=responsibility['notify'])
                        node_responsibilities[canonical.as_key()] = canonical

            children = None
            if 'values' in role_data:
                direct = False
                children = role_data.get('values')
            elif 'roles' in role_data:
                direct = True
                children = role_data.get('roles')
            elif 'children' in role_data:
                direct = True
                children = role_data.get('children')

            name, name_en = get_lang_pair(role_data.get('name', ''))
            description, description_en = get_lang_pair(role_data.get('help', ''))

            children_nodes = ()
            if children:
                children_nodes = self.normalize_inner(children, direct=direct)
            canonical_node = CanonicalNode(
                slug=role_data['slug'],
                name=name,
                name_en=name_en,
                description=description,
                description_en=description_en,
                is_exclusive=role_data.get('is_exclusive', False),
                is_public=role_data.get('visibility', True),
                unique_id=role_data.get('unique_id', ''),
                comment_required=role_data.get('comment_required', False),
                set=role_data.get('set', ''),
                fields=node_fields,
                aliases=node_aliases,
                responsibilities=node_responsibilities,
                children=children_nodes,
                hash=None,
            )
            canonicals.append(canonical_node)
        return tuple(canonicals)

    @staticmethod
    def as_bool(value):
        if isinstance(value, str) and value.lower() in ('false', '0'):
            value = False
        else:
            value = bool(value)
        return value
