# coding: utf-8


import abc
import argparse
import logging

from django.core.exceptions import ValidationError
from django.core.management.base import BaseCommand
from django.forms import fields, models as modelfields
from django.utils import translation

from django_tools_log_context.profiler import execution_profiler

from idm.utils.command_timestamp import timestamp_context
from idm.utils.i18n import set_translation
from idm.utils.lock import lock


class ArgType(object):
    def __init__(self, field_class, *args, **kwargs):
        self.field = field_class(*args, **kwargs)

    def __call__(self, value):
        with translation.override('en'):
            try:
                return self.field.clean(value)
            except ValidationError as e:
                raise argparse.ArgumentTypeError(str(e))


class IdmBaseCommand(BaseCommand, metaclass=abc.ABCMeta):
    """Базовый класс для argparse-based команд. Регистрирует новые типы"""

    USE_LOCK = True
    intranet_only = False
    b2b_only = False

    _logger: logging.Logger = None

    @property
    def logger(self) -> logging.Logger:
        if self._logger is None:
            self._logger = logging.getLogger(self.__module__)
        return self._logger

    def add_arguments(self, parser):
        from idm.core.models import System

        parser.register(
            'type', 'system', ArgType(modelfields.ModelChoiceField, System.objects.all(), to_field_name='slug')
        )
        parser.register('type', 'datetime', ArgType(fields.DateTimeField))
        parser.register('type', 'int', ArgType(fields.IntegerField))

        parser.add_argument(
            '--use-block-lock',
            dest='use_block_lock',
            action='store_true',
            help='Использовать блокирующий лок',
            default=False,
        )

    @abc.abstractmethod
    def idm_handle(self, *args, **options):
        pass

    @property
    def lock_name(self):
        return '{}.{}'.format(self.__class__.__module__, self.__class__.__name__)

    @property
    def command_name(self):
        return 'commands.{}'.format(self.__class__.__module__.split('.')[-1])

    def stamped_run(self, *args, **options):
        threshold_for_extra_logs = 60 * 1000
        with timestamp_context(self.command_name, self.logger):
            with execution_profiler(self.command_name, 'management command', threshold_for_extra_logs):
                self.idm_handle(*args, **options)

    @set_translation
    def handle(self, *args, **options):
        if self.b2b_only:
            self.logger.warning('Command `%s` work only in b2b' % self.__class__.__module__)
            return
        if not self.USE_LOCK:
            return self.stamped_run(*args, **options)

        with lock(self.lock_name, block=options['use_block_lock']) as locked:
            if not locked:
                return

            self.stamped_run(*args, **options)
