# -*- coding: utf-8 -*-


import logging


from idm.core.management.base import IdmBaseCommand
from idm.core.models import System
from idm.sync import everysync
from idm.utils.i18n import set_translation
from idm.utils.lock import lock
from idm.utils.replication import use_slave

log = logging.getLogger(__name__)


class Command(IdmBaseCommand):
    USE_LOCK = False

    help = """Проверяет изменения ролей систем и для ветвей,
    помеченных для автоматического обновления, производит обновление"""

    def add_arguments(self, parser):
        super().add_arguments(parser)

        parser.add_argument(
            '--system',
            type='system',
            help='Обрабатывать только конкретную систему.',
        )

    @use_slave()
    @set_translation
    def idm_handle(self, *args, **options):
        log.info('Loading systems with auto update flag from DB')

        systems = System.objects.get_operational()
        system = options.get('system')
        if system is not None:
            systems = systems.filter(pk=system.pk)
        else:
            systems = systems.ready_for_sync()
        with lock('idm_auto_update_roles', block=False) as acquired:
            if not acquired:
                return
            for system in systems:
                everysync.sync_roles_and_nodes(
                    system,
                    steps=everysync.SYNC_NODES_ONLY,
                    force_nodes=False,
                    requester=None,
                    block=False,
                )
