# -*- coding: utf-8 -*-
import logging

from django.core.management import call_command
from django.conf import settings
from django.db.models import NOT_PROVIDED
from django.utils.timezone import datetime, timedelta

from idm.core.management.base import IdmBaseCommand
from idm.core.models import System
from idm.sync import everysync
from idm.utils.i18n import set_translation

log = logging.getLogger(__name__)


def _system_fits_schedule(scatter_id):
    """
    равномерно распределить системы в по рабочему дню
    и сказать попадает ли система `scatter_id` в интервал по текущему времени
    :param scatter_id: некое число, уникальное для системы, по которому идет деление систем на интервалы
    """
    interval = settings.IDM_SCHEDULED_SYSTEM_SYNC_INTERVAL
    now = datetime.now()
    if now.weekday() >= 5:
        return False  # Не синхронизируем на выходных
    today = now.replace(hour=0, minute=0, second=0, microsecond=0)

    # добавляем по 5 минут чтобы вызовы крона не попадали на границы интервалов
    safe_offset = timedelta(minutes=5)
    day_start = today.replace(hour=settings.IDM_SCHEDULED_SYSTEM_SYNC_BEGIN_HOUR) - safe_offset
    day_end = today.replace(hour=settings.IDM_SCHEDULED_SYSTEM_SYNC_END_HOUR) - safe_offset

    total_intervals = int((day_end - day_start).total_seconds() // interval)
    current_interval = int((now - day_start).total_seconds() // interval)

    if not (0 <= current_interval < total_intervals):
        return False
    return scatter_id % total_intervals == current_interval


def _filter_systems_for_schedule(systems):
    systems = systems.order_by('-active_nodes_count', 'pk')
    for i, system in enumerate(systems):
        if _system_fits_schedule(i):
            yield system


class Command(IdmBaseCommand):
    help = '''Сверяем роли пользователей во всех доступных системах, а затем разрешаем все неразрешенные
    (кроме неизвестных пользователей) расхождения.
    Если есть расхождения, то отчет отправляется на специальную рассылку.'''

    def add_arguments(self, parser):
        super().add_arguments(parser)

        parser.add_argument(
            '--system',
            type='system',
            help='Обрабатывать только конкретную систему',
        )
        parser.add_argument(
            '--threshold',
            type='int',
            help='После скольких расхождений считать систему сломанной.'
        ),
        parser.add_argument(
            '--check-only',
            action='store_true',
            dest='check_only',
            default=False,
            help='Только проверить расхождения, не разрешая их'
        )
        parser.add_argument(
            '--resolve-only',
            action='store_true',
            dest='resolve_only',
            default=False,
            help='Только разрешить уже имеющиеся расхождения, не проверяя систему вновь',
        ),
        parser.add_argument(
            '--log-time',
            action='store_true',
            dest='log_check_time',
            default=False,
            help='Записывать время выполнения команды (если сверка прошла успешно) в metainfo'
        ),
        parser.add_argument(
            '--report-unchecked',
            action='store_true',
            dest='report_unchecked',
            default=False,
            help='Записы'
        )
        parser.add_argument(
            '--daily-schedule-interval',
            action='store_true',
            dest='daily_schedule_interval',
            default=False,
            help='Запустить синхронизацию только для тех систем,'
                 'которые должны быть синхронизированны в данный интервал рабочего дня'
        )

    @set_translation
    def idm_handle(self, *args, **options):
        threshold = options.get('threshold') or NOT_PROVIDED

        systems = System.objects.operational().order_by('slug')
        if options.get('system'):
            systems = systems.filter(pk=options['system'].pk)
        if options.get('daily_schedule_interval'):
            systems = _filter_systems_for_schedule(systems)

        for system in systems:
            if system.plugin_type == 'dumb':
                steps = {'deprive_nodes'}
            else:
                steps = {'sync_nodes', 'deprive_nodes'}
                if not options.get('check_only', False):
                    steps.add('resolve_inconsistencies')
                if not options.get('resolve_only', False):
                    steps.add('check_inconsistencies')
                    steps.add('report_inconsistencies')
                if options.get('report_unchecked'):
                    steps.add('report_unchecked')

            everysync.sync_roles_and_nodes(
                system,
                force_roles=False,
                from_web=False,
                steps=steps,
                threshold=threshold,
                requester=None,
                log_check_time=options.get('log_check_time', False),
                block=True,
            )
