# coding: utf-8



import waffle

from django.db.models import F, Q
from django.utils import timezone

from idm.core.management.base import IdmBaseCommand
from idm.core.models import RoleNode
from idm.sync.intrasearch import IntrasearchFetcher


class Command(IdmBaseCommand):
    help = 'Проверяет slug_path у измененных/созданных через апи узлов'

    def _can_find_in_suggest(self, fetcher, node):
        scope = node.parent.parent.self_path if node.parent.parent else '/'
        return bool(fetcher.search(text=node.slug_path, filters={'s_parent_path': scope}))

    def _get_slug_path(self, node):
        slugs = ['', node.slug]
        while node.parent:
            slugs.append(node.parent.slug)
            node = node.parent

        return '/'.join(reversed(slugs))

    def check_slug_paths(self):
        pushed_nodes = (
            RoleNode.objects
            .filter(
                Q(slug_path_ok_after_push_at__isnull=True)| Q(slug_path_ok_after_push_at__lt=F('pushed_at')),
                pushed_at__isnull=False
            )
        )
        for node in pushed_nodes:
            if node.slug_path == self._get_slug_path(node):
                node.slug_path_ok_after_push_at = timezone.now()
                node.save(update_fields=['slug_path_ok_after_push_at'])

        moved_nodes = (
            RoleNode.objects
            .filter(
                Q(slug_path_ok_after_move_at__isnull=True)| Q(slug_path_ok_after_move_at__lt=F('moved_at')),
                moved_at__isnull=False
            )
        )
        for node in moved_nodes:
            if node.slug_path == self._get_slug_path(node):
                node.slug_path_ok_after_move_at = timezone.now()
                node.save(update_fields=['slug_path_ok_after_move_at'])

    def check_suggest(self):
        fetcher = IntrasearchFetcher('idm_rolenodes')

        moved_nodes = (
            RoleNode.objects
            .filter(
                Q(suggest_ok_after_move_at=None) | Q(suggest_ok_after_move_at__lt=F('moved_at')),
                moved_at__isnull=False,
                slug_path_ok_after_move_at__gt=F('moved_at'),
            )
            .select_related('parent__parent')
        )
        for node in moved_nodes:
            if self._can_find_in_suggest(fetcher, node):
                node.suggest_ok_after_move_at = timezone.now()
                node.save(update_fields=['suggest_ok_after_move_at'])

        pushed_nodes = (
            RoleNode.objects
            .filter(
                Q(suggest_ok_after_push_at=None) | Q(suggest_ok_after_push_at__lt=F('pushed_at')),
                pushed_at__isnull=False,
                slug_path_ok_after_push_at__gt=F('pushed_at'),
            )
            .select_related('parent__parent')
        )

        for node in pushed_nodes:
            if self._can_find_in_suggest(fetcher, node):
                node.suggest_ok_after_push_at = timezone.now()
                node.save(update_fields=['suggest_ok_after_push_at'])

    def idm_handle(self, *args, **kwargs):
        if not waffle.switch_is_active('check_slug_paths'):
            return

        self.check_slug_paths()
        self.check_suggest()
